/** Provides a dataflow configuration to reason about improper validation of code-specified size used for array construction. */

import java
private import semmle.code.java.security.internal.ArraySizing
private import semmle.code.java.dataflow.TaintTracking

/**
 * A dataflow configuration to reason about improper validation of code-specified size used for array construction.
 */
module BoundedFlowSourceConfig implements DataFlow::ConfigSig {
  predicate isSource(DataFlow::Node source) {
    source instanceof BoundedFlowSource and
    // There is not a fixed lower bound which is greater than zero.
    not source.(BoundedFlowSource).lowerBound() > 0
  }

  predicate isSink(DataFlow::Node sink) {
    any(CheckableArrayAccess caa).canThrowOutOfBoundsDueToEmptyArray(sink.asExpr(), _)
  }

  predicate observeDiffInformedIncrementalMode() { any() }

  Location getASelectedSinkLocation(DataFlow::Node sink) {
    exists(ArrayCreationExpr arrayCreation, CheckableArrayAccess arrayAccess |
      result = [arrayCreation, arrayAccess.getIndexExpr()].getLocation() and
      arrayAccess.canThrowOutOfBoundsDueToEmptyArray(sink.asExpr(), arrayCreation)
    )
  }
}

/**
 * Dataflow flow for improper validation of code-specified size used for array construction.
 */
module BoundedFlowSourceFlow = DataFlow::Global<BoundedFlowSourceConfig>;
