/** Provides classes and predicates to reason about path injection vulnerabilities. */

import rust
private import codeql.rust.controlflow.BasicBlocks
private import codeql.rust.controlflow.ControlFlowGraph
private import codeql.rust.dataflow.DataFlow
private import codeql.rust.dataflow.TaintTracking
private import codeql.rust.Concepts
private import codeql.rust.dataflow.internal.DataFlowImpl
private import codeql.rust.controlflow.ControlFlowGraph as Cfg

/**
 * Provides default sources, sinks and barriers for detecting path injection
 * vulnerabilities, as well as extension points for adding your own.
 */
module TaintedPath {
  /**
   * A data flow source for path injection vulnerabilities.
   */
  abstract class Source extends DataFlow::Node { }

  /**
   * A data flow sink for path injection vulnerabilities.
   */
  abstract class Sink extends QuerySink::Range {
    override string getSinkType() { result = "TaintedPath" }
  }

  /**
   * A barrier for path injection vulnerabilities.
   */
  abstract class Barrier extends DataFlow::Node { }

  /**
   * A sanitizer guard for path-traversal vulnerabilities.
   */
  class SanitizerGuard extends DataFlow::Node {
    SanitizerGuard() { this = DataFlow::BarrierGuard<sanitizerGuard/3>::getABarrierNode() }
  }

  /**
   * An active threat-model source, considered as a flow source.
   */
  private class ActiveThreatModelSourceAsSource extends Source, ActiveThreatModelSource { }

  /** A sink for path-injection from model data. */
  private class ModelsAsDataSinks extends Sink {
    ModelsAsDataSinks() { sinkNode(this, "path-injection") }
  }
}

private predicate sanitizerGuard(AstNode g, Expr e, boolean branch) {
  g.(SanitizerGuard::Range).checks(e, branch)
}

/** Provides a class for modeling new path safety checks. */
module SanitizerGuard {
  /** A data-flow node that checks that a path is safe to access. */
  abstract class Range extends AstNode {
    /** Holds if this guard validates `e` upon evaluating to `branch`. */
    abstract predicate checks(Expr e, boolean branch);
  }
}

/**
 * A check of the form `!strings.Contains(nd, "..")`, considered as a sanitizer guard for
 * path traversal.
 */
private class DotDotCheck extends SanitizerGuard::Range, MethodCallExpr {
  DotDotCheck() {
    this.getStaticTarget().(Addressable).getCanonicalPath() =
      ["<alloc::string::String>::contains", "<core::str>::contains"] and
    this.getArg(0).(LiteralExpr).getTextValue() = ["\"..\"", "\"../\"", "\"..\\\""]
  }

  override predicate checks(Expr e, boolean branch) {
    e = this.getReceiver() and
    branch = false
  }
}
