/*
 * Scala.js (https://www.scala-js.org/)
 *
 * Copyright EPFL.
 *
 * Licensed under Apache License 2.0
 * (https://www.apache.org/licenses/LICENSE-2.0).
 *
 * See the NOTICE file distributed with this work for
 * additional information regarding copyright ownership.
 */

package org.scalajs.testsuite.javalib.lang

import java.math.BigInteger
import java.util.SplittableRandom

import org.junit.Test
import org.junit.Assert._

import org.scalajs.testsuite.utils.AssertThrows.assertThrows

class MathTestOnJDK11 {

  @noinline
  private def hideFromOptimizer(x: Int): Int = x

  @Test def multiplyExactLongInt(): Unit = {
    for (n <- Seq(Long.MinValue, -1L, 0L, 1L, Long.MaxValue)) {
      assertEquals(0L, Math.multiplyExact(n, 0))
      assertEquals(n, Math.multiplyExact(n, 1))
    }
    for (n <- Seq(Int.MinValue, -1, 0, 1, Int.MaxValue)) {
      assertEquals(0L, Math.multiplyExact(0L, n))
      assertEquals(n.toLong, Math.multiplyExact(1L, n))
    }
    assertEquals(Long.MaxValue, Math.multiplyExact(-9223372036854775807L, -1))
    assertEquals(2147483648L, Math.multiplyExact(-1L, Int.MinValue))
    assertEquals(31284307708346368L, Math.multiplyExact(-14567891L, Int.MinValue))
    assertEquals(9223372036854775806L, Math.multiplyExact(4611686018427387903L, 2))
    assertEquals(922337202L, Math.multiplyExact(2L, 461168601))
    assertEquals(Long.MinValue, Math.multiplyExact(4611686018427387904L, -2))
    assertEquals(-4294967294L, Math.multiplyExact(-2L, Int.MaxValue))
    assertEquals(-6415938107894138L, Math.multiplyExact(-2987654L, Int.MaxValue))

    assertThrows(classOf[ArithmeticException], Math.multiplyExact(Long.MinValue, -1))
    assertThrows(classOf[ArithmeticException], Math.multiplyExact(-12345678910L, Int.MinValue))
    assertThrows(classOf[ArithmeticException], Math.multiplyExact(Long.MinValue, Int.MinValue))
    assertThrows(classOf[ArithmeticException], Math.multiplyExact(Long.MaxValue, Int.MaxValue))
    assertThrows(classOf[ArithmeticException], Math.multiplyExact(Long.MinValue, Int.MaxValue))
    assertThrows(classOf[ArithmeticException], Math.multiplyExact(Long.MaxValue, Int.MinValue))
    assertThrows(classOf[ArithmeticException], Math.multiplyExact(4611686018427387904L, 2))
    assertThrows(classOf[ArithmeticException], Math.multiplyExact(29876541321L, 461168601))
    assertThrows(classOf[ArithmeticException], Math.multiplyExact(4611686018427387905L, -2))
    assertThrows(classOf[ArithmeticException], Math.multiplyExact(-29876541321L, 461168601))
  }

  @Test def testMultiplyFull(): Unit = {
    @inline def test(expected: Long, x: Int, y: Int): Unit = {
      assertEquals(expected, Math.multiplyFull(x, y))
      assertEquals(expected, Math.multiplyFull(x, hideFromOptimizer(y)))
      assertEquals(expected, Math.multiplyFull(hideFromOptimizer(x), y))
      assertEquals(expected, Math.multiplyFull(hideFromOptimizer(x), hideFromOptimizer(y)))
    }

    test(2641928036408725662L, 1942041231, 1360387202)
    test(54843908448922272L, 1565939409, 35023008)
    test(510471553407128558L, 1283300489, 397780222)
    test(-1211162085735907941L, -1990140693, 608581137)
    test(-1197265696701533712L, -584098468, 2049766884)
    test(203152587796496856L, -1809591416, -112264341)
    test(-1869763755321108598L, 1235591906, -1513253483)
    test(-737954189546644064L, 675415792, -1092592442)
    test(-2570904460570261986L, 1639253754, -1568338309)
    test(1106623967126000400L, 2088029790, 529984760)
    test(1407516248272451352L, -869881054, -1618055988)
    test(-2120367337662071940L, -1558894530, 1360173698)
    test(-1464086284066637244L, -1417313902, 1033000722)
    test(36729253163312334L, -1673852034, -21942951)
    test(-3197007331876781046L, 1876799847, -1703435418)
    test(461794994386945009L, -246001091, -1877207099)
    test(-1206231192496917804L, 867896526, -1389832954)
    test(-1739671893103255929L, -1083992841, 1604873969)
    test(-409626127116780624L, 240101424, -1706054551)
    test(-3083566560548370936L, -1568530113, 1965895672)
    test(-1205028798380605000L, -1201743532, 1002733750)
    test(-1328689065035027168L, 929349664, -1429697687)
    test(-124212693522020684L, 80893862, -1535502082)
    test(-82341860111074830L, -243230690, 338534007)
    test(-846837059701860202L, 1959770926, -432110227)
    test(335728245390354432L, 506816728, 662425344)
    test(745294755971022170L, 1521993302, 489683335)
    test(-2370525755201631608L, 2023520366, -1171485988)
    test(-1039854583047715776L, 593162592, -1753068378)
    test(-152985384388127808L, -635946432, 240563319)
    test(-678107568956539050L, 649113254, -1044667575)
    test(-3064094283703186444L, -1890896836, 1620444979)
    test(1240687269228318870L, -1080325230, -1148438669)
    test(-46551523496333580L, 27167878, -1713476610)
    test(-2500430606368427103L, 2023288183, -1235825241)
    test(92963399778762084L, 896198732, 103730787)
    test(2469065794894324667L, 2105111101, 1172890967)
    test(172558569988357136L, -142945148, -1207166332)
    test(335684786634110970L, -1647598405, -203741874)
    test(2406859843746696240L, 2049365815, 1174441296)
    test(3100973294006114952L, 1991928152, 1556769651)
    test(-335912134649077352L, 866240524, -387781598)
    test(84303320581066207L, 75666091, 1114149277)
    test(-2623126349572207976L, 1426933667, -1838295928)
    test(59139945163750590L, 149344270, 395997417)
    test(-105764175098643999L, 68726447, -1538915217)
    test(8595303129864000L, 726092025, 11837760)
    test(-2958527843471399088L, 1536412078, -1925608296)
    test(1532625839159904477L, 867021537, 1767690621)
    test(384402376484481316L, 1207235521, 318415396)
    test(-219376614576542698L, 1816299166, -120782203)
    test(-672138807810988440L, 531516745, -1264567512)
    test(-193351903065245331L, 170858169, -1131651499)
    test(71263251057597648L, 51058196, 1395725988)
    test(-774312974742971385L, 1958551603, -395349795)
    test(-1846593638370672048L, 1190143097, -1551572784)
    test(240083094242536384L, 1404614968, 170924488)
    test(-130950827889833280L, -115480554, 1133964320)
    test(128954457719585228L, 735993884, 175211317)
    test(364779990580792000L, -668489125, -545678272)
    test(107252402494512045L, 759517757, 141211185)
    test(3038084150893069044L, -1924640913, -1578519988)
    test(760804294233336624L, -728394552, -1044494762)
    test(1171051779605774913L, 848233701, 1380576813)
    test(-1805862307837393080L, -1385644986, 1303264780)
    test(172227703288618734L, -104999826, -1640266559)
    test(150448013961014407L, 163398103, 920745169)
    test(-671469201380991232L, 650262784, -1032612073)
    test(-1325861126942924945L, -1773644581, 747534845)
    test(987406376890116568L, -1626507773, -607071416)
    test(2918138947401192144L, 1695881208, 1720721318)
    test(-2590993826910153940L, -1397240042, 1854365570)
    test(954644624447419276L, -1516139806, -629654746)
    test(407510452326678620L, -384747652, -1059162935)
    test(149866317537821404L, 1530355444, 97929091)
    test(922044716091910632L, 968149268, 952378674)
    test(-3508732521573808284L, 1825364562, -1922209182)
    test(1701723136959404304L, 894776752, 1901841027)
    test(-2435876799625512705L, -1276062909, 1908900245)
    test(-516933170985379201L, 657063047, -786732983)
    test(123334479976750576L, 313765817, 393078128)
    test(-1072624004420456775L, -894199299, 1199535725)
    test(301682711612188737L, 330918981, 911651277)
    test(1790992996470651507L, -1115945231, -1604911197)
    test(-2750453268538140155L, 1878389719, -1464261245)
    test(758285757353272504L, 1259684942, 601964612)
    test(-218581674312137400L, -161533394, 1353167100)
    test(-1824007072461951836L, -1244277844, 1465916219)
    test(-92753167730460334L, -65368843, 1418920138)
    test(-2326636630979491248L, 1124395877, -2069232624)
    test(-7380586257943446L, 29715454, -248375349)
    test(31319707234597638L, 491995506, 63658523)
    test(-1196559502630778250L, -1752963990, 682592175)
    test(166065559841839548L, -911521074, -182185102)
    test(-1222260378510810100L, 1071539812, -1140657925)
    test(57800571165871464L, -257569032, -224408077)
    test(332444627169725608L, 1247224172, 266547614)
    test(217903869180130650L, 1069161915, 203808110)
    test(920425054266935850L, -901689546, -1020778225)
    test(-507632632656614388L, 864632142, -587108214)
  }

  @Test def testMultiplyHigh(): Unit = {
    def test(expected: Long, x: Long, y: Long): Unit =
      assertEquals(expected, Math.multiplyHigh(x, y))

    test(-2514789262281153376L, 8217931296694472096L, -5644933286224084859L)
    test(-298247406641127011L, -8034902747807161194L, 684724352445702293L)
    test(242644198957550459L, 717019025263929004L, 6242505821226454837L)
    test(-1089698470915011537L, -7558081430876177893L, 2659588811568490384L)
    test(138675986327040026L, 2362930226177876193L, 1082605148727562445L)
    test(-1260260349245855816L, -3350308785473442797L, 6938972380570262589L)
    test(-1799534229489533301L, -4097805274432763180L, 8100811327075225922L)
    test(437623091041087696L, -2968271773754119013L, -2719670493975918294L)
    test(-107841114219899514L, 2013609532543228156L, -987936043452088475L)
    test(2757621741022067854L, -7005993850636185311L, -7260803191272031988L)
    test(-187671345159116030L, 1781219534362173574L, -1943570237881252419L)
    test(-515018730942796014L, 6085558843030314089L, -1561141543105626636L)
    test(-119091959391883575L, 7423442237814967910L, -295935339127164155L)
    test(18351865713513547L, -1886460125362775846L, -179453657960126825L)
    test(3928100041033091765L, 8449838094261471293L, 8575389888485029447L)
    test(-7404756889594137L, -89549316594063561L, 1525345591296625693L)
    test(714591873345926311L, -2929853068304815970L, -4499165349746322236L)
    test(1305977852854305585L, -5568549492657237090L, -4326268312655360053L)
    test(-2435010516398991446L, 6443930667478151719L, -6970592660082469124L)
    test(2031324595328562735L, 5390460907312723801L, 6951413911530987604L)
    test(34713245667458599L, -535353692461820541L, -1196118319182197181L)
    test(255381044848343425L, -3176530727082196631L, -1483048388428836603L)
    test(6566871520624982L, -33326351213089011L, -3634883324950494373L)
    test(156130078476475485L, 687410849583778615L, 4189767446364284457L)
    test(1647679448547038188L, 4460502251200507739L, 6814102850116870938L)
    test(-2241611115434343963L, 5633894511267143863L, -7339581257068946568L)
    test(-93572860194426351L, -1075368508503119813L, 1605137764964203383L)
    test(1663347345126188661L, -6330756750592024018L, -4846710115399342760L)
    test(-1686630202076061136L, 5124142056960069542L, -6071813649745693328L)
    test(728105493712673843L, -8079843401135830331L, -1662306437683128283L)
    test(-2030727779883712688L, 4452689522888653156L, -8412963770845872378L)
    test(734253555387491804L, 5835084770836409518L, 2321232330529258387L)
    test(2018627311798804222L, -7211950082779933827L, -5163250018863045382L)
    test(-1244560006523295051L, -7326211205612788508L, 3133690700470219958L)
    test(-492070935033321215L, 1614944457187625808L, -5620692751550184667L)
    test(319340972880203566L, 2310036532484690677L, 2550090059672932009L)
    test(1766280783448332865L, 5949345770128658249L, 5476590340096838859L)
    test(2757208297958468913L, -5707089944199929572L, -8911987777945981523L)
    test(408328069441815717L, 1242541635079749093L, 6062028975489127199L)
    test(-77985829287979398L, -7943526433115400350L, 181101510313367840L)
    test(-230121117022373017L, -780391911062895469L, 5439555807140802418L)
    test(2588662639521587653L, 7451684432618227097L, 6408268846625040081L)
    test(861249002493118404L, 1744344496585548181L, 9107856827493957233L)
    test(-2703044944335540474L, 8052570526613861366L, -6192106997771248181L)
    test(-2975059248415970510L, 6503508572335523474L, -8438546047759521035L)
    test(-370291189062632935L, -8722964233277178137L, 783067156383574516L)
    test(-90473002639507852L, 852694261922564555L, -1957245873225555126L)
    test(-218977334338454381L, -1819563432425194345L, 2219993418476586419L)
    test(-1087231185918604076L, -2941838679159182506L, 6817462690146034563L)
    test(-1170480051005916145L, -2771463765488827700L, 7790665067735548924L)
    test(-371145713487913188L, 3224241917397787909L, -2123423169279885562L)
    test(-502492608136209963L, 1568228348895174267L, -5910716094215359887L)
    test(1445926343733049503L, -7706328512722939071L, -3461133686196008644L)
    test(-1374053009197983052L, -8787832166727089323L, 2884306814637966447L)
    test(-1910150305525172307L, 8663815092401732543L, -4067036686787486282L)
    test(2074971709256543740L, 8092193156887080609L, 4730049238662438083L)
    test(953725989108917020L, 8492699833366153401L, 2071560232049848145L)
    test(334989155711573307L, 1093268576921704206L, 5652279186765632978L)
    test(129011196343964709L, 1000276763122669782L, 2379178052852915387L)
    test(239042793587178901L, 3208737625070847213L, 1374235525371105170L)
    test(127809344420152430L, -7696730067895344868L, -306320508313194466L)
    test(-2506455997163955037L, -5731747797284935902L, 8066641092198683254L)
    test(3016086034985660469L, -6992699346126002928L, -7956436339922591224L)
    test(-1527917483534567268L, -8938885845855254814L, 3153089016969294968L)
    test(-1268939936756528050L, 5537112727075101653L, -4227439716695399205L)
    test(-37535014067603004L, -8605247800544091240L, 80462389271855887L)
    test(-2710920384572235679L, -7926242046619125682L, 6309125338878172023L)
    test(-3331830886924716794L, 6823617049086893513L, -9007163096323738999L)
    test(1854911433578401793L, -4644835313936852982L, -7366693150982113934L)
    test(-3840461794042836575L, 8006480391435326631L, -8848334396141248546L)
    test(-1212641710132993432L, -7017377545321262459L, 3187699555205380404L)
    test(946047090630044138L, -5829622550331878687L, -2993588077419595837L)
    test(3518955178043574292L, -7909090733489625033L, -8207424565425867851L)
    test(1231895337081111773L, 2841977238766797132L, 7996002817598962425L)
    test(-1649686524869089287L, -3558405071306300052L, 8551962049372852642L)
    test(1156466789444347220L, -8077807627762096372L, -2640945152160624636L)
    test(-284428196958678125L, 7604654143237097972L, -689942508603024688L)
    test(24530734973246035L, -4976536915346383672L, -90929133590073966L)
    test(915668791878818L, -4915702564252847L, -3436153355352311231L)
    test(-59487608720960501L, 2234272329433906652L, -491145452224512365L)
    test(-935777346233643464L, 2234022931260640741L, -7726888105936443458L)
    test(-539196324963981948L, 1233384294780865907L, -8064328899098291942L)
    test(-302740552339519239L, 1652272762436229815L, -3379936785683182277L)
    test(-1602328337662720444L, -5891195966699023422L, 5017273391344774367L)
    test(1971437877011804292L, 6123334000940359947L, 5939021122948580484L)
    test(3518273874050862283L, -7935043146462869940L, -8178997459486413381L)
    test(989386049294028022L, 3631504400505165814L, 5025727419987895939L)
    test(1075600553777136761L, 8162668046881939535L, 2430740540606242760L)
    test(555876997051543592L, -1422006546765159905L, -7211022146415941068L)
    test(1442987791832810570L, 3172003226122803882L, 8391676993961733131L)
    test(122174343239443206L, 592078109511582332L, 3806455273225175653L)
    test(-555975358284841098L, -2610695041141095892L, 3928430928909536969L)
    test(1217820260754824228L, -2566343358431797989L, -8753629401971345682L)
    test(-843540703271762806L, 2010390971620435041L, -7740076278033066915L)
    test(28227414827282063L, 1691814723551530731L, 307778322255183098L)
    test(-3487482743675782331L, 8885183126228404590L, -7240447464066348779L)
    test(-641218088086423374L, -5793475349478143447L, 2041673650588512538L)
    test(491218135799199820L, -3483174304311045377L, -2601470510458659970L)
    test(-61083956648009538L, -331097881159246733L, 3403223576515274855L)
    test(-1760654512150512675L, -6642702867806073297L, 4889326503714183951L)
  }

}
