/*
 * Scala.js (https://www.scala-js.org/)
 *
 * Copyright EPFL.
 *
 * Licensed under Apache License 2.0
 * (https://www.apache.org/licenses/LICENSE-2.0).
 *
 * See the NOTICE file distributed with this work for
 * additional information regarding copyright ownership.
 */

package org.scalajs.testsuite.compiler

import org.junit.Test
import org.junit.Assert._
import org.junit.Assume._

import org.scalajs.testsuite.utils.AssertExtensions.assertExactEquals
import org.scalajs.testsuite.utils.AssertThrows.assertThrows
import org.scalajs.testsuite.utils.Platform._

class LongTest {
  import LongTest._

  // Helpers

  @noinline def hideFromOptimizer(x: Long): Long = x

  @noinline def hideDoubleFromOptimizer(x: Double): Double = x

  @noinline def hideAnyFromOptimizer(x: Any): Any = x

  // Common values

  def MaxVal: Long = 0x7fffffffffffffffL
  def MinVal: Long = 0x8000000000000000L
  def IntMaxVal: Long = 0x000000007fffffffL
  def IntMinVal: Long = 0xffffffff80000000L
  def IntMaxValPlus1: Long = 0x0000000080000000L
  def IntMinValMinus1: Long = 0xffffffff7fffffffL
  def MaxSafeDouble: Long = 0x001fffffffffffffL
  def TwoPow53: Long = 0x0020000000000000L
  def MinSafeDouble: Long = 0xffe0000000000001L
  def NegTwoPow53: Long = 0xffe0000000000000L

  // Tests

  @Test def sanityOfEqualityTests(): Unit = {
    assertEquals(1958505087099L, 1958505087099L)
    assertEquals(528280977864L, 528280977864L)

    assertNotEquals(17179869307L, 1958505087099L)
    assertNotEquals(17179869307L, 1958505087099L)
    assertNotEquals(1958505086977L, 1958505087099L)
    assertNotEquals(1958505086977L, 1958505087099L)
    assertNotEquals(123L, 1958505087099L)
  }

  @Test def equalsAny(): Unit = {
    @inline def inlineCallEquals(lhs: Long, rhs: Any): Boolean =
      lhs.asInstanceOf[AnyRef].equals(rhs.asInstanceOf[AnyRef])

    @inline def test(expected: Boolean, lhs: Long, rhs: Any): Unit = {
      assertEquals(expected, inlineCallEquals(lhs, rhs))
      assertEquals(expected, inlineCallEquals(hideFromOptimizer(lhs), rhs))
      assertEquals(expected, inlineCallEquals(lhs, hideAnyFromOptimizer(rhs)))
      assertEquals(expected,
          inlineCallEquals(hideFromOptimizer(lhs), hideAnyFromOptimizer(rhs)))
    }

    test(false, 0L, 0)
    test(false, 0L, null)

    test(true, 0L, 0L)
    test(true, 1958505087099L, 1958505087099L)
    test(true, 1962800054149L, 1962800054149L)
    test(true, -1954210119803L, -1954210119803L)

    test(false, 1958505087099L, 1962800054149L)
    test(false, 1958505087099L, -1958505086853L)
    test(false, -1954210119803L, -1958505086853L)
    test(false, -1954210119803L, 1962800054149L)
  }

  @Test def literals(): Unit = {
    assertEquals(105L, 5L + 100L)
    assertEquals(2147483651L, 2147483649L + 2L)
    assertEquals(-8589934592L, -2147483648L * 4)
    assertEquals(-18014398509482040L, 4503599627370510L * (-4))
  }

  @Test def unaryOps(): Unit = {
    val x = 10L
    assertEquals(-10L, -x)
    val y = 5L
    assertEquals(-5L, -y)
    assertEquals(5L, +y)
    assertEquals(-6L, ~y)
  }

  @Test def binaryOps(): Unit = {
    assertEquals(25F, 5L * 5F, 0F)
    assertEquals(1F, 5L % 4F, 0F)
    assertEquals(20F, 5F * 4L, 0F)
  }

  @Test def shifts_Issue622(): Unit = {
    def l(x: Long): Long = x
    def i(x: Int): Int = x

    assertEquals(268435455L, l(-7L) >>> 100L)
    assertEquals(-1L, l(-7L) >> 100L)
    assertEquals(-1L, l(-7L) >> 100)
    assertEquals(268435455L, l(-7L) >>> 100)
    assertEquals(-481036337152L, l(-7L) << 100L)
    assertEquals(-481036337152L, l(-7L) << 100)
    assertEquals(481036337152L, l(7L) << 100L)
    assertEquals(549755813888L, l(8L) << 100L)
    assertEquals(1152921504606846975L, l(-7L) >>> 4)

    assertEquals(112, i(7) << 100)
    assertEquals(-1, i(-7) >> 100)
    assertEquals(268435455, i(-7) >>> 100)
    assertEquals(-5, i(-65) >> 100)
    assertEquals(-5, i(-65) >> 4)
  }

  @Test def toLongConversions(): Unit = {
    // Byte
    assertEquals(112L, 112.toByte.toLong)
    // Short
    assertEquals(-10L, (-10).toShort.toLong)
    // Char
    assertEquals(65L, 'A'.toLong)
    // Int
    assertEquals(5L, 5.toLong)
    // Long
    assertEquals(10L, 10L.toLong)
    // Float
    assertEquals(100000L, 100000.6f.toLong)
    // Double
    assertEquals(100000L, 100000.6.toLong)
  }

  @Test def testHashCodeLiterals(): Unit = {
    assertEquals(0, 0L.hashCode())
    assertEquals(55, 55L.hashCode())
    assertEquals(11, (-12L).hashCode())
    assertEquals(10006548, 10006548L.hashCode())
    assertEquals(1098747, (-1098748L).hashCode())

    assertEquals(-825638905, 613354684553L.hashCode())
    assertEquals(1910653900, 9863155567412L.hashCode())
    assertEquals(1735398658, 3632147899696541255L.hashCode())
    assertEquals(-1689438124, 7632147899696541255L.hashCode())
  }

  @Test def hashHash(): Unit = {
    assertEquals(0, 0L.##)
    assertEquals(55, 55L.##)
    assertEquals(-12, (-12L).##)
    assertEquals(10006548, 10006548L.##)
    assertEquals(-1098748, (-1098748L).##)

    assertEquals(1910653900, 9863155567412L.##)
    assertEquals(1735398658, 3632147899696541255L.##)

    assertEquals(-825638905, 613354684553L.##)
    assertEquals(-1689438124, 7632147899696541255L.##)
  }

  @Test def hashHashInCaseClasses(): Unit = {
    if (scalaVersion.startsWith("2.12.")) {
      assertEquals(-1669410282, HashTestBox(0L).##)
      assertEquals(-1561146018, HashTestBox(55L).##)
      assertEquals(-1266055417, HashTestBox(-12L).##)
      assertEquals(-1383472436, HashTestBox(10006548L).##)
      assertEquals(1748124846, HashTestBox(-1098748L).##)

      assertEquals(1291324266, HashTestBox(9863155567412L).##)
      assertEquals(-450677189, HashTestBox(3632147899696541255L).##)

      assertEquals(259268522, HashTestBox(1461126709984L).##)
      assertEquals(818387364, HashTestBox(1L).##)
    } else {
      assertEquals(1445817443, HashTestBox(0L).##)
      assertEquals(536512430, HashTestBox(55L).##)
      assertEquals(2131034006, HashTestBox(-12L).##)
      assertEquals(713468002, HashTestBox(10006548L).##)
      assertEquals(-1926941956, HashTestBox(-1098748L).##)

      assertEquals(1150870245, HashTestBox(9863155567412L).##)
      assertEquals(-1713893803, HashTestBox(3632147899696541255L).##)

      assertEquals(-1901418683, HashTestBox(1461126709984L).##)
      assertEquals(-491089524, HashTestBox(1L).##)
    }
  }

  @Test def concatWithString(): Unit = {
    val x = 20L
    assertEquals("asdf520hello", "asdf" + 5L + x + "hello")
    assertEquals("20hello", x + "hello")
  }

  @Test def stringToLong(): Unit = {
    assertEquals(45678901234567890L, "45678901234567890".toLong)
  }

  @Test def asInstanceOf(): Unit = {
    val dyn:  Any  = 5L
    val stat: Long = 5L

    assertEquals(5L, stat.asInstanceOf[Long])
    // models current scala behavior. See SI-1448
    assertEquals(5, stat.asInstanceOf[Int])

    assertTrue(stat.isInstanceOf[Long])
    assertFalse(stat.isInstanceOf[Int])

    assertEquals(5L, dyn.asInstanceOf[Long])

    assertTrue(dyn.isInstanceOf[Long])
    assertFalse(dyn.isInstanceOf[Int])
  }

  @Test def asInstanceOfNegative(): Unit = {
    assumeTrue("Assumed compliant asInstanceOf", hasCompliantAsInstanceOfs)

    val dyn: Any = 5L
    assertThrows(classOf[Exception], dyn.asInstanceOf[Int])
  }

  @Test def compareOtherNumericTypes(): Unit = {
    assertTrue(5L == 5)
    assertTrue(5 == 5L)
    assertTrue(4 != 5L)
    assertTrue('A' == 65L)
  }

  @Test def testHashCode(): Unit = {
    @inline def test(expected: Int, x: Long): Unit = {
      assertEquals(expected, x.hashCode())
      assertEquals(expected, hideFromOptimizer(x).hashCode())
    }

    test(0, 0L)
    test(0, -1L)
    test(55, 55L)
    test(11, -12L)
    test(10006548, 10006548L)
    test(1098747, -1098748L)

    test(957662195, 1982825597567543955L)
    test(-1075860794, 1406034298302928504L)
    test(1425294575, -1179570764218296756L)
    test(-1863811248, -5924750827952573997L)
    test(-881942797, -2395914305612636591L)
    test(548587254, 2447885248525657180L)
    test(-1328999812, -5504896584370634811L)
    test(-1756412154, 384199524456576104L)
    test(-529144798, 7888567912398337690L)
    test(-1163319584, 5735688670530266260L)
    test(2070477069, 4433036395378750992L)
    test(-1718642695, 1179554006416034279L)
    test(260982265, -8357730361210493571L)
    test(-385578983, 7428661341186121784L)
    test(-1362397169, 2543158706518910289L)
    test(1419211160, 3229416646666240204L)
    test(-1851816270, -4010623825090125741L)
    test(112959880, -7968980127741386941L)
    test(1715333902, -1854772489226326735L)
    test(-453690224, -3970683778215415628L)
    test(-1503679197, 126638565233673273L)
    test(1950154296, 8797042038419333400L)
    test(2037562473, -6972503273726145185L)
    test(220707473, 8280642475761957604L)
    test(1902658020, 5228410264741169803L)
    test(840583449, -3278640449314249480L)
    test(2065572837, -3875486092154527221L)
    test(407536450, 5288520647426579136L)
    test(-1678479110, 7046653027410693485L)
    test(-1558558486, -4024377053108452722L)
    test(-110470482, -838396561058221973L)
    test(992932874, 5129444237846349393L)
    test(2035378556, -983133572388201403L)
    test(542449527, -6104122336102002076L)
    test(-1824846728, -415158756909382857L)
    test(-985103709, -1969167716381843770L)
    test(37361715, -5476315081547833279L)
    test(-1555729529, -7743071034335664532L)
    test(1534845437, -7538261814076156116L)
    test(-715250396, -1429728134370512654L)
    test(2003953821, 2162071222722023631L)
    test(1631287431, 5863241404849132153L)
    test(-1393125048, 4881214556208830350L)
    test(926193137, 3242961417161524683L)
    test(1141998463, -7308151016669202399L)
    test(480895538, 4641424446555440323L)
    test(-849143869, -4721253045988906339L)
    test(-1840233445, -1282181253417368854L)
    test(47538111, -5095937255712569425L)
    test(540301593, 1165015579268918886L)
    test(1903332829, 3548911248465907239L)
    test(-1325859168, -7033666394241918194L)
    test(-1476869146, -3604991509767603020L)
    test(84316181, -8970981391004714015L)
    test(524038724, -8506822844828777483L)
    test(-794988445, 4933058946008665754L)
    test(-1285356617, 1378197176240526512L)
    test(1441713710, 8713431403779497372L)
    test(365800340, -8905685952204829093L)
    test(2130603708, -6817584868623549422L)
    test(-1414171289, -5367476554073402908L)
    test(-262714124, 8262747073669783643L)
    test(158195454, -2248665858028667730L)
    test(50128093, -3976075402628120415L)
    test(-825145129, 2776265820355301338L)
    test(-1344834498, -4107878840944388661L)
    test(-103814738, 7956081985265747911L)
    test(-1354282241, 3715014944073787543L)
    test(1408148925, -5636885620037294119L)
    test(1910019874, 8158721670792678656L)
    test(1877620608, -1851180655167832448L)
    test(-378358620, -861494542715981853L)
    test(492633155, -7380175794367584908L)
    test(-1581166836, 3854049825653594857L)
    test(174532880, 859390584786911807L)
    test(-629188646, -4435899092567076195L)
    test(2139225425, -1325122976028439811L)
    test(200043623, 7646723521622766304L)
    test(1992690082, 3282098032132903783L)
    test(754072038, 785689457701776768L)
    test(-139359279, -1890968847650888181L)
    test(-1669264515, 4844892846454312738L)
    test(25583899, -1626066900596724257L)
    test(1822592670, 442937131460406887L)
    test(1468680630, -3825110455431449608L)
    test(2103231504, -1301560781171381939L)
    test(1159389820, 1138240706348922264L)
    test(776506096, 2985427124709938391L)
    test(-1303579924, 6802445941252806246L)
    test(1108767081, 2456207273236644585L)
    test(-1101969936, -7706583586557791489L)
    test(-1022615009, -1692614123490213261L)
    test(-1221218252, 4616796007159011253L)
    test(410005178, 1349693816546475482L)
    test(1180107886, -3820218088529535091L)
    test(425308062, -2793089286027256951L)
    test(1727927187, 749334892891357070L)
    test(-759140792, -3566029960059386643L)
    test(1698140938, -9022872570059308487L)
    test(512144461, -4192565604012017786L)
  }

  @Test def toStringTest(): Unit = {
    @inline def test(expected: String, x: Long): Unit = {
      assertEquals(expected, x.toString())
      assertEquals(expected, hideFromOptimizer(x).toString())
    }

    test("0", 0L)
    test("1", 1L)
    test("-1", -1L)
    test("2147483647", IntMaxVal)
    test("2147483648", IntMaxValPlus1)
    test("-2147483648", IntMinVal)
    test("-2147483649", IntMinValMinus1)
    test("999999999", 999999999L)
    test("1000000000", 1000000000L)
    test("9007199254740991", MaxSafeDouble)
    test("9007199254740992", TwoPow53)
    test("-9007199254740991", MinSafeDouble)
    test("-9007199254740992", NegTwoPow53)

    test("-86922", -86922L)
    test("0", 0L)
    test("-21874015", -21874015L)
    test("-2098921896914", -2098921896914L)
    test("80985205273168", 80985205273168L)
    test("-12451732102972849", -12451732102972849L)
    test("3350", 3350L)
    test("-92511590195450", -92511590195450L)
    test("-2", -2L)
    test("446248293253325286", 446248293253325286L)
    test("499596119314678396", 499596119314678396L)
    test("-3205893", -3205893L)
    test("-88762100292970", -88762100292970L)
    test("-1278004", -1278004L)
    test("-1", -1L)
    test("-305393", -305393L)
    test("-2", -2L)
    test("80295210784300943", 80295210784300943L)
    test("5", 5L)
    test("21", 21L)
    test("64", 64L)
    test("39146094", 39146094L)
    test("-1725731", -1725731L)
    test("-768047304243556260", -768047304243556260L)
    test("-2726923242838", -2726923242838L)
    test("-1781092907033", -1781092907033L)
    test("-213275", -213275L)
    test("7662405832810", 7662405832810L)
    test("-154157877107", -154157877107L)
    test("-929963900939521435", -929963900939521435L)
    test("-6872", -6872L)
    test("31842553544728", 31842553544728L)
    test("567569520305426", 567569520305426L)
    test("19649016", 19649016L)
    test("-1349346", -1349346L)
    test("9479824673588660", 9479824673588660L)
    test("3521781", 3521781L)
    test("1740", 1740L)
    test("0", 0L)
    test("92834698468", 92834698468L)
    test("-80139798970631138", -80139798970631138L)
    test("30058", 30058L)
    test("-611022189550002", -611022189550002L)
    test("514941281681226", 514941281681226L)
    test("2454759250363", 2454759250363L)
    test("14860137468144958", 14860137468144958L)
    test("-79255", -79255L)
    test("2290122305310796", 2290122305310796L)
    test("-755641947927852310", -755641947927852310L)
    test("-2621852156570472370", -2621852156570472370L)
    test("-37956135735", -37956135735L)
    test("853219", 853219L)
    test("901", 901L)
    test("4385596303898", 4385596303898L)
    test("-972597865", -972597865L)
    test("-8057379", -8057379L)
    test("-14968", -14968L)
    test("-98204964", -98204964L)
    test("335479", 335479L)
    test("-429441918886", -429441918886L)
    test("9798741", 9798741L)
    test("135908509698671494", 135908509698671494L)
    test("-141095409221912371", -141095409221912371L)
    test("-9040837797787104", -9040837797787104L)
    test("-889", -889L)
    test("3222082994", 3222082994L)
    test("-1454853", -1454853L)
    test("547641844425", 547641844425L)
    test("2528132853", 2528132853L)
    test("242", 242L)
    test("-1655763891", -1655763891L)
    test("82", 82L)
    test("-120254181", -120254181L)
    test("-210088", -210088L)
    test("-2", -2L)
    test("250255458324299", 250255458324299L)
    test("-100656997", -100656997L)
    test("-24097181761", -24097181761L)
    test("206088", 206088L)
    test("-593", -593L)
    test("-99542049", -99542049L)
    test("421501", 421501L)
    test("-2", -2L)
    test("-101", -101L)
    test("3", 3L)
    test("14967492854", 14967492854L)
    test("-1528445803513883", -1528445803513883L)
    test("26760588095306", 26760588095306L)
    test("12452686330472", 12452686330472L)
    test("-130630407827875", -130630407827875L)
    test("-10281777615", -10281777615L)
    test("-90497242609445", -90497242609445L)
    test("-13935178716929", -13935178716929L)
    test("-11308540", -11308540L)
    test("545166", 545166L)
    test("-1043705339124703", -1043705339124703L)
    test("510", 510L)
    test("-2485453027", -2485453027L)
    test("-15103", -15103L)
    test("-168776672025670194", -168776672025670194L)
  }

  @Test def toByte(): Unit = {
    @inline def test(expected: Byte, x: Long): Unit = {
      assertEquals(expected, x.toByte)
      assertEquals(expected, hideFromOptimizer(x).toByte)
    }

    test(0, 0L)
    test(-1, -1L)
    test(0x98.toByte, 0x76543210fedcba98L)

    test(102, -5348390297197908122L)
    test(77, 5977507544004298317L)
    test(-47, 6514359347096730577L)
    test(-22, 4993772461838941418L)
    test(-113, 6430368993139692943L)
    test(-84, -9085781647198927956L)
    test(33, 6306810273394300449L)
    test(113, -2210139158093195919L)
    test(12, 8144711790963478540L)
    test(86, -5600967788150588842L)
    test(-36, -6380820517237296420L)
    test(88, 634449370258994520L)
    test(107, 335338151813720939L)
    test(61, 1067478847354529853L)
    test(99, -6293272425705388957L)
    test(-76, 7140156260973973940L)
    test(0, 5600320537509350144L)
    test(-69, 3631108045842532795L)
    test(26, -7844071689366422502L)
    test(64, -1401566392023965632L)
  }

  @Test def toShort(): Unit = {
    @inline def test(expected: Short, x: Long): Unit = {
      assertEquals(expected, x.toShort)
      assertEquals(expected, hideFromOptimizer(x).toShort)
    }

    test(0, 0L)
    test(-1, -1L)
    test(0xba98.toShort, 0x76543210fedcba98L)

    test(-670, -3488924154499629726L)
    test(-15861, -7040498320163487221L)
    test(9963, -8181870893638998293L)
    test(18394, -7321494144773503014L)
    test(-7956, -3480431557828484884L)
    test(21453, 4105103149183292365L)
    test(22979, -3021078359861798461L)
    test(8452, -4765875786069171964L)
    test(-26563, -3748769590109235139L)
    test(-5754, -7644892824540223098L)
    test(11796, -2109015304010125804L)
    test(18020, -1573587201743436188L)
    test(3768, 4144850305644236472L)
    test(-4579, 6246434176940961309L)
    test(-29102, -3393569066699878830L)
    test(30020, 7196258688511997252L)
    test(-13051, -1470416838456849147L)
    test(-2720, 3209606815575962976L)
    test(4726, 7336600601467032182L)
    test(-16878, -8385845263650669038L)
  }

  @Test def toInt(): Unit = {
    @inline def test(expected: Int, x: Long): Unit = {
      assertEquals(expected, x.toInt)
      assertEquals(expected, hideFromOptimizer(x).toInt)
    }

    test(0, 0L)
    test(-1, -1L)
    test(0xfedcba98, 0x76543210fedcba98L)

    test(-1869423218, -23694035066364530L)
    test(450655357, -2240222333751233411L)
    test(-596464514, 2703727000802208894L)
    test(1668957409, 5287278019205547233L)
    test(-313016061, 1217656393840443651L)
    test(-406779255, 5967093472329534089L)
    test(-1125423893, -1876561510140254997L)
    test(1491309031, 4073352392381534695L)
    test(360542935, -4440368487118048553L)
    test(178673916, -8786934223313938180L)
    test(-1167644863, 3172689046147639105L)
    test(-1852739075, 4083832281038418429L)
    test(-1965326912, 7279924828375317952L)
    test(-141857741, -5143473252658877389L)
    test(-938893686, 7574413823307259530L)
    test(-1178638558, 1284483777703143202L)
    test(-1296424902, -7277623459310854086L)
    test(204387309, -1033964890421808147L)
    test(-942136876, -2265025955962019372L)
    test(-1703892744, 1031594642500528376L)
  }

  @Test def toLong(): Unit = {
    @inline def test(expected: Long, x: Long): Unit = {
      assertEquals(expected, x.toLong)
      assertEquals(expected, hideFromOptimizer(x).toLong)
    }

    test(0L, 0L)
    test(-1L, -1L)
    test(0x76543210fedcba98L, 0x76543210fedcba98L)

    test(6907420169189163269L, 6907420169189163269L)
    test(-6558938415102325809L, -6558938415102325809L)
    test(-7633462319206780754L, -7633462319206780754L)
    test(-4051533910437546682L, -4051533910437546682L)
    test(-3890339056676572253L, -3890339056676572253L)
    test(-3091543614186826784L, -3091543614186826784L)
    test(2806266116723834799L, 2806266116723834799L)
    test(-1741184441450532748L, -1741184441450532748L)
    test(3395924718030703835L, 3395924718030703835L)
    test(-7712245542997911283L, -7712245542997911283L)
    test(-2751064647855401745L, -2751064647855401745L)
    test(5225909624054208018L, 5225909624054208018L)
    test(1334025594846136121L, 1334025594846136121L)
    test(-1574909139329823322L, -1574909139329823322L)
    test(-9142211941778525044L, -9142211941778525044L)
    test(-5517402195275269807L, -5517402195275269807L)
    test(7612683537409046411L, 7612683537409046411L)
    test(-2955859733488660001L, -2955859733488660001L)
    test(462084382441397543L, 462084382441397543L)
    test(8801656334077465992L, 8801656334077465992L)
  }

  @Test def toFloat(): Unit = {
    @inline def test(expected: Float, x: Long): Unit = {
      assertExactEquals(expected, x.toFloat)
      assertExactEquals(expected, hideFromOptimizer(x).toFloat)
    }

    test(0, 0L)
    test(-1, -1L)

    test(9.223372E18f, MaxVal)
    test(-9.223372E18f, MinVal)

    test(4.7971489E18f, 4797148756851199825L)
    test(-2.24047663E18f, -2240476588964556043L)
    test(4.59211416E18f, 4592114141414587341L)
    test(3.38942079E18f, 3389420784882550191L)
    test(-6.8076878E18f, -6807687928853852340L)
    test(7.4159717E18f, 7415971944732783122L)
    test(-1.85275997E18f, -1852759910613995193L)
    test(5.7344188E18f, 5734418636597215975L)
    test(3.20410168E18f, 3204101606352738628L)
    test(-7.2310311E18f, -7231031002290834816L)
    test(7.7151362E18f, 7715136160435444567L)
    test(1.41365268E18f, 1413652736819778775L)
    test(-3.03197918E18f, -3031979149645678356L)
    test(-4.04287594E18f, -4042876010967900169L)
    test(-7.8204678E18f, -7820467788284364348L)
    test(-5.9733025E18f, -5973302693141309798L)
    test(1.1261721E18f, 1126172094624944445L)
    test(4.00884963E18f, 4008849511969667347L)
    test(-1.43511611E18f, -1435116153438497835L)
    test(3.81415059E18f, 3814150611446505582L)

    // #4466 Long values that are close to Float midpoints

    test(3.7930952e16f, 0x86c2007fffffffL)
    test(3.7930952e16f, 0x86c20080000000L) // midpoint, even is downwards
    test(3.7930956e16f, 0x86c20080000001L) // this is the value from #4466

    test(3.7930965e16f, 0x86c2037fffffffL)
    test(3.7930969e16f, 0x86c20380000000L) // midpoint, even is upwards
    test(3.7930969e16f, 0x86c20380000001L)

    test(-3.7930952e16f, -0x86c2007fffffffL)
    test(-3.7930952e16f, -0x86c20080000000L) // midpoint, even is downwards
    test(-3.7930956e16f, -0x86c20080000001L)

    test(-3.7930965e16f, -0x86c2037fffffffL)
    test(-3.7930969e16f, -0x86c20380000000L) // midpoint, even is upwards
    test(-3.7930969e16f, -0x86c20380000001L)
  }

  @Test def toDouble(): Unit = {
    @inline def test(expected: Double, x: Long): Unit = {
      assertExactEquals(expected, x.toDouble)
      assertExactEquals(expected, hideFromOptimizer(x).toDouble)
    }

    test(0, 0L)
    test(-1, -1L)

    test(9.223372036854776E18, MaxVal)
    test(-9.223372036854776E18, MinVal)

    test(3.4240179834317537E18, 3424017983431753968L)
    test(8.5596043411285968E16, 85596043411285973L)
    test(-3.1630346897289943E18, -3163034689728994087L)
    test(-4.4847682439933604E18, -4484768243993360332L)
    test(-6.4014772289576371E17, -640147722895763706L)
    test(-1.76968119148756736E18, -1769681191487567328L)
    test(-8.5606671350959739E18, -8560667135095973729L)
    test(-9.0403963253949932E18, -9040396325394992652L)
    test(-6.4988752582247977E18, -6498875258224797887L)
    test(-7.7788492399114394E17, -777884923991143875L)
    test(7.6357174849871442E18, 7635717484987144190L)
    test(1.25338659134517658E18, 1253386591345176663L)
    test(-3.1910241505692349E18, -3191024150569234776L)
    test(7.4216510087652332E18, 7421651008765232983L)
    test(-8.189046896086654E18, -8189046896086653617L)
    test(6.8316272807487539E18, 6831627280748754219L)
    test(-8.0611115909320561E18, -8061111590932056057L)
    test(1.7127521901359959E18, 1712752190135995904L)
    test(-6.4442523492577423E18, -6444252349257742098L)
    test(-1.71264450938175027E18, -1712644509381750301L)
  }

  @Test def fromDouble(): Unit = {
    @inline def test(expected: Long, x: Double): Unit = {
      assertEquals(expected, x.toLong)
      assertEquals(expected, hideDoubleFromOptimizer(x).toLong)
    }

    val twoPow63 = 9.223372036854776E18
    val twoPow63NextUp = 9.223372036854778E18
    val twoPow63NextDown = 9.2233720368547748E18

    // Specials
    test(0L, 0.0)
    test(0L, -0.0)
    test(0L, Double.NaN)
    test(MaxVal, Double.PositiveInfinity)
    test(MinVal, Double.NegativeInfinity)

    // Corner cases
    test(0L, Double.MinPositiveValue)
    test(0L, -Double.MinPositiveValue)
    test(MaxVal, twoPow63)
    test(MaxVal, twoPow63NextUp)
    test(9223372036854774784L, twoPow63NextDown)
    test(MinVal, -twoPow63)
    test(MinVal, -twoPow63NextUp)
    test(-9223372036854774784L, -twoPow63NextDown)

    // Absolute value too big
    test(MaxVal, 1.5623101234432471E19)
    test(MaxVal, 1.0425697303244048E19)
    test(MaxVal, 1.500625248806836E19)
    test(MinVal, -1.5623101234432471E19)
    test(MinVal, -1.0425697303244048E19)
    test(MinVal, -1.500625248806836E19)

    // Normal cases
    test(-6048920506403873L, -6.048920506403873E15)
    test(8503743119053764L, 8.503743119053764E15)
    test(3939528382405885L, 3.939528382405885E15)
    test(-4008535239847255L, -4.008535239847255E15)
    test(-761635408727443L, -7.61635408727443E14)
    test(2424447379938472L, 2.424447379938472E15)
    test(-4648424796281871L, -4.648424796281871E15)
    test(8533360864252241L, 8.533360864252241E15)
    test(-2774715761463054L, -2.774715761463054E15)
    test(3500774757068786L, 3.500774757068786E15)
    test(-4456998776411743L, -4.456998776411743E15)
    test(5090705064274507L, 5.090705064274507E15)
    test(-3304575013039063L, -3.304575013039063E15)
    test(1770193656876L, 1.770193656876E12)
    test(-2176598598697488L, -2.176598598697488E15)
    test(4101884566378912L, 4.101884566378912E15)
    test(4085180300766691L, 4.085180300766691E15)
    test(-4657426781904383L, -4.657426781904383E15)
    test(2850079490584325L, 2.850079490584325E15)
    test(6087335263699874L, 6.087335263699874E15)
    test(-6713475274360568L, -6.713475274360568E15)
    test(4162543436756133L, 4.162543436756133E15)
    test(-7955791959986676L, -7.955791959986676E15)
    test(5231358553020274L, 5.231358553020274E15)
    test(1499979916805674L, 1.499979916805674E15)
    test(5082547854412290L, 5.08254785441229E15)
    test(2809489845729537L, 2.809489845729537E15)
    test(4877349929065833L, 4.877349929065833E15)
    test(-6846998949739153L, -6.846998949739153E15)
    test(6198980017388729L, 6.198980017388729E15)
    test(4948567844262856L, 4.948567844262856E15)
    test(-3707206656862071L, -3.707206656862071E15)
    test(-8665283507887641L, -8.665283507887641E15)
    test(-2862150709693603L, -2.862150709693603E15)
    test(1587366173692588L, 1.587366173692588E15)
    test(332903144317135L, 3.32903144317135E14)
    test(5902580156722507L, 5.902580156722507E15)
    test(-1001746319375376L, -1.001746319375376E15)
    test(-519938829196113L, -5.19938829196113E14)
    test(-8130332101524554L, -8.130332101524554E15)
    test(-1769686613392113L, -1.769686613392113E15)
    test(-7292607053441567L, -7.292607053441567E15)
    test(2667276401109539L, 2.667276401109539E15)
  }

  @Test def comparisons(): Unit = {
    @inline def testInner(x: Long, y: Long, expected: Int): Unit = {
      assertEquals(expected, x.compareTo(y).signum)
      assertEquals(expected, x.compareTo(y: java.lang.Long).signum)
      assertEquals(expected == 0, x.equals(y))
      assertEquals(expected != 0, !x.equals(y))
      assertEquals(expected < 0, x < y)
      assertEquals(expected <= 0, x <= y)
      assertEquals(expected > 0, x > y)
      assertEquals(expected >= 0, x >= y)
    }

    @inline def test(x: Long, y: Long, expected: Int): Unit = {
      testInner(x, y, expected)
      testInner(hideFromOptimizer(x), y, expected)
      testInner(x, hideFromOptimizer(y), expected)
      testInner(hideFromOptimizer(x), hideFromOptimizer(y), expected)
    }

    test(0L, 0L, 0)
    test(0L, 1L, -1)
    test(0L, -1L, 1)
    test(MaxVal, MinVal, 1)
    test(MinVal, MaxVal, -1)

    // Positive and negative numbers requiring lo to be compared via unsigned
    test(0x654789ab87654321L, 0x654789ab12345678L, 1)
    test(0x89abcdef87654321L, 0x89abcdef12345678L, 1)

    // Whitebox corner cases
    test(4294967295L, 0L, 1)
    test(0L, 4294967295L, -1)

    test(-8092415883884355943L, -8092415883884355943L, 0)
    test(-2031213215071472067L, -2031213215071472067L, 0)
    test(5283240141261796897L, 5283240141261796897L, 0)
    test(7116056094667264990L, 7116056094667264990L, 0)
    test(-3116536807772461648L, -3116536807772461648L, 0)
    test(-6248629176470511083L, -6248629176470511083L, 0)
    test(4454865470354617868L, 4454865470354617868L, 0)
    test(999717031143899307L, 999717031143899307L, 0)
    test(-5297317869524400407L, -5297317869524400407L, 0)
    test(8939068183804817614L, 8939068183804817614L, 0)

    test(3236335092840668602L, -444219516364298554L, 1)
    test(-4943613816686756195L, -7742547864832660642L, 1)
    test(8029769392898531897L, 4652159919261674509L, 1)
    test(5796873813047518622L, 6289828457790219503L, -1)
    test(-4146252130540320321L, 229162157128594750L, -1)
    test(1666874288313754998L, -7814084014110160137L, 1)
    test(3712093048452543434L, 4733495962840491197L, -1)
    test(-945689136431988108L, 173720678004198908L, -1)
    test(1942384584602846367L, -4109783896492426928L, 1)
    test(1481147942356422128L, 2728187183550782760L, -1)
    test(8772007041528428997L, -9186971037150267738L, 1)
    test(8245726731682071232L, 5163318085402121929L, 1)
    test(1299197553427533446L, 6802087801757261262L, -1)
    test(3305926949416808899L, -2540594350510012206L, 1)
    test(-2961552791416065935L, -6969814195378031172L, 1)
    test(4088157660605840986L, 7194896515898720601L, -1)
    test(-1405157718104570917L, 6578730965739299783L, -1)
    test(3272049947913267945L, 8706977984604794647L, -1)
    test(542774473992239193L, -5153605571562040274L, 1)
    test(-3021233129684732581L, -6793741445281541653L, 1)
    test(-3173969257539621561L, 8203748404252766862L, -1)
    test(5803462967220487068L, 5590091268406426455L, 1)
    test(5773238099961046596L, 4776817781713990253L, 1)
    test(1563566144471605276L, -7443854257335427327L, 1)
    test(-5267335731847770188L, 5984710255758911411L, -1)
    test(7824450174170580562L, -6006761146392923340L, 1)
    test(-6310031417822260419L, 378610261756145452L, -1)
    test(-399600969910632649L, -130497809780743843L, -1)
    test(-4718563483403518902L, 5540460094616151815L, -1)
    test(3635009130414443975L, 4486809672346912398L, -1)
    test(-6995802727856870261L, 6569166150278248282L, -1)
    test(6968791371002309508L, -8193565738888199170L, 1)
    test(-1517460660612243341L, -8851020320296654181L, 1)
    test(-1291017114648111885L, 7754360291495616390L, -1)
    test(2390342010645922745L, 3110449275385222696L, -1)
    test(4802823317634197020L, -3390019797807795163L, 1)
    test(-6688721861751417385L, -2010281469393235264L, -1)
    test(6834502861598033678L, 4612059587200791166L, 1)
    test(-7600578759410557298L, 6343221779008750856L, -1)
    test(-2603046777061840807L, -4567951219930893995L, 1)
    test(5011577782742676064L, 257378676128675336L, 1)
    test(-8818761229247524974L, 7361903928218635798L, -1)
    test(-460239612252633323L, -7662854947344368725L, 1)
    test(2977807727662937042L, -7549105702637036210L, 1)
    test(415628984375646263L, -5181382544518532798L, 1)
    test(-8808201391150486985L, 2766868173945056047L, -1)
    test(-2438752987770741867L, -2682801229352239102L, 1)
    test(-9039176582481370526L, 5109370859012131857L, -1)
    test(-6518810838185736764L, 612030854711736193L, -1)
    test(-723222763675766024L, -8452156148778795797L, 1)
  }

  @Test def bitwiseNot(): Unit = {
    @inline def test(expected: Long, x: Long): Unit = {
      assertEquals(expected, ~x)
      assertEquals(expected, ~hideFromOptimizer(x))
    }

    test(1406386578883106454L, -1406386578883106455L)
    test(-5065753424737328870L, 5065753424737328869L)
    test(2498180912133439506L, -2498180912133439507L)
    test(-3417617645097176974L, 3417617645097176973L)
    test(4952400515267264813L, -4952400515267264814L)
    test(4149430339078675315L, -4149430339078675316L)
    test(4961809918450635435L, -4961809918450635436L)
    test(-295808714778363857L, 295808714778363856L)
    test(-5033087994080563510L, 5033087994080563509L)
    test(2335191956353907186L, -2335191956353907187L)
    test(2976543784068121798L, -2976543784068121799L)
    test(5774257430242417862L, -5774257430242417863L)
    test(-5203158850856805292L, 5203158850856805291L)
    test(2656867406598406382L, -2656867406598406383L)
    test(349944556951922984L, -349944556951922985L)
    test(-2235069048512104852L, 2235069048512104851L)
    test(-6659120388296241551L, 6659120388296241550L)
    test(-3040957425825662851L, 3040957425825662850L)
    test(-5775077234374864294L, 5775077234374864293L)
    test(-1518767499246577383L, 1518767499246577382L)
    test(1070100640767033738L, -1070100640767033739L)
    test(8895696740996914294L, -8895696740996914295L)
    test(1914207089783931554L, -1914207089783931555L)
    test(1793754405624530053L, -1793754405624530054L)
    test(-4727596867237275197L, 4727596867237275196L)
    test(-5311066053511426959L, 5311066053511426958L)
    test(7619154394655448072L, -7619154394655448073L)
    test(5250967210084652069L, -5250967210084652070L)
    test(-5455364456349674746L, 5455364456349674745L)
    test(1019609221300419525L, -1019609221300419526L)
    test(2187298499813494865L, -2187298499813494866L)
    test(7089445246515851057L, -7089445246515851058L)
    test(-117530700556341000L, 117530700556340999L)
    test(7264034611998876040L, -7264034611998876041L)
    test(7750189185721635078L, -7750189185721635079L)
    test(-7930464363410409312L, 7930464363410409311L)
    test(-3724992682063679742L, 3724992682063679741L)
    test(-69347497633896303L, 69347497633896302L)
    test(-7081869313756020883L, 7081869313756020882L)
    test(-260714627593142486L, 260714627593142485L)
    test(1185242357839654984L, -1185242357839654985L)
    test(-7564605668187644040L, 7564605668187644039L)
    test(3871856848880622734L, -3871856848880622735L)
    test(-4817126385920070332L, 4817126385920070331L)
    test(-5888158719107453004L, 5888158719107453003L)
    test(-6109337908490803053L, 6109337908490803052L)
    test(1976010683499657518L, -1976010683499657519L)
    test(-4432126610911889114L, 4432126610911889113L)
    test(-462096843480338453L, 462096843480338452L)
    test(-241839956263802374L, 241839956263802373L)
  }

  @Test def bitwiseOr(): Unit = {
    @inline def test(expected: Long, x: Long, y: Long): Unit = {
      assertEquals(expected, x | y)
      assertEquals(expected, hideFromOptimizer(x) | y)
      assertEquals(expected, x | hideFromOptimizer(y))
      assertEquals(expected, hideFromOptimizer(x) | hideFromOptimizer(y))
    }

    test(-2613546261823747L, -732254410524747223L, 5345832810278072061L)
    test(3421574493437913976L, 2696479001637514040L, 797647112316668528L)
    test(-16042512477593622L, -664842653033835350L, -448952131140493374L)
    test(5830854331665341311L, 1182857968182411900L, 5827951603767122247L)
    test(-720603522827034697L, -1920249875726639821L, 5811721741070616501L)
    test(-2429989451465175L, -326992767357722080L, -5866127430977143287L)
    test(-45248344289067025L, -2353906758366184474L, -1055227015112063061L)
    test(-4698666246569598983L, -4734697660499845319L, 1621457430914915536L)
    test(-24814331965472769L, -926314086088462360L, 2279900522349328671L)
    test(8059190707467844254L, 2852111230804619804L, 7842960731354904726L)
    test(-1299871242251165713L, -3683824728419169311L, 2699284952865513135L)
    test(-41870100751729666L, -5232550147682886938L, -2428804309254984770L)
    test(6908433860711458460L, 5393524540374819456L, 6890418190755046556L)
    test(7993562706145703421L, 7416377889934169588L, 1010726668980695177L)
    test(-172295982489308161L, -4856286452619001099L, 7032617688688324942L)
    test(-2918992351416288789L, 6065686119411425387L, -2931663545932777054L)
    test(8322018733155337727L, 5794053415639762293L, 7141908358983730638L)
    test(-4755872684732145997L, -4826261879547622223L, 1584870117983689363L)
    test(-2476417429357203461L, -2548561902270470214L, 1549343018945998411L)
    test(-369653621121128705L, -5557800598580943118L, 5211256272336404557L)
    test(-3171660142335247466L, -4351855033137776256L, 1289125874441159318L)
    test(-787216413499655748L, 6130308206832544156L, -2017009453335512668L)
    test(7926297957957107629L, 3012135534784600329L, 5528131121954895525L)
    test(-1153215787209601037L, 3281699202371666403L, -1765934039245778991L)
    test(6881359415537745819L, 1688635789095255962L, 6787565517611350803L)
    test(-8648116384209080578L, -9202397995633793986L, -8668383956993675538L)
    test(-602670364721666L, 7483112994760842748L, -385664398910673506L)
    test(-292888044845940995L, -292891458105855768L, -2746857536523755907L)
    test(-4977890735616730628L, -7329350491424339716L, -6277252349998458456L)
    test(4503388486656715234L, 4490713934143219808L, 17741664735021538L)
    test(-5701559780695884293L, -5701741286170118677L, -6914436005175818056L)
    test(-5779831442641846291L, 993300921744289769L, -5797856736963307292L)
    test(-290763737917315074L, 5473740978342430494L, -5078653238533283082L)
    test(-74318466972713413L, -233070474449197557L, -2686500410628990928L)
    test(-6920485765606572033L, 1811874579410020463L, -8145465210131030094L)
    test(-288372222960273450L, 7691970788769154832L, -1053498197837811242L)
    test(9217689610436018175L, 9118186006662675386L, 3416982267728807389L)
    test(9145472665524008399L, 8845664388695744903L, 7253942521130268106L)
    test(8520800324501585391L, 2612004993903957345L, 5920089518940241038L)
    test(-5334523139102147867L, -6564568800883844411L, 1564742708242051684L)
    test(-6940364003006747649L, -6987881135588046647L, -8021804063776707714L)
    test(-6801716370022401L, -170695486873386145L, -1736202733831922463L)
    test(-216493289859239969L, 615458773225571547L, -218758568622478452L)
    test(-374093511683146753L, 3084633618707954610L, -1116221274237416993L)
    test(-2387587339420727361L, -4405345383298137980L, 6636200699812185915L)
    test(-1731705165849379329L, 4875035904673097819L, -6596017243635173915L)
    test(-147501890251260097L, -148100032629917899L, 7535477669170072367L)
    test(-686808179565166593L, -8776722885123410241L, -840635388947440167L)
    test(-2924806642048485385L, -2933995819876272797L, -4386233589193955660L)
    test(-154249953701857396L, -2171950689813499516L, 9063633311846033288L)
  }

  @Test def bitwiseAnd(): Unit = {
    @inline def test(expected: Long, x: Long, y: Long): Unit = {
      assertEquals(expected, x & y)
      assertEquals(expected, hideFromOptimizer(x) & y)
      assertEquals(expected, x & hideFromOptimizer(y))
      assertEquals(expected, hideFromOptimizer(x) & hideFromOptimizer(y))
    }

    test(76866864474772480L, 1518159482761016584L, -6641824821105304521L)
    test(-3457554706992136192L, -3157747983614630634L, -2808454669606774239L)
    test(-9212111935053496320L, -8111098352606606816L, -1101013866591026742L)
    test(-8067906238548507614L, -437628452145933277L, -8062623626330200390L)
    test(1306063832401432868L, 6028164543958998829L, 1902368571311640996L)
    test(290484375182639114L, 291752186448215071L, 9010342492458418378L)
    test(153197360447488148L, 7955027074777874868L, 153779005941156511L)
    test(2381559777949401217L, 8443545652655876545L, 2526276136490124973L)
    test(1152924260097024184L, 1395569193552412350L, 4071661456730611897L)
    test(36028870340188164L, -6218303295630779004L, 1624191667329907236L)
    test(51952143485652240L, -8504253278454231208L, 1637268985052167571L)
    test(2309581386439540738L, 3575110474069786663L, -5877919068377462574L)
    test(-8080434026952257519L, -2315168695265372909L, -8071108514578657035L)
    test(6276493755819008L, 8112792678725019438L, 657187380755735744L)
    test(217449317261139968L, -2062987362041764220L, 1992149119552090440L)
    test(-4176234055483758504L, -3598644602753160866L, -1861119886376276775L)
    test(4704223293901654016L, 5857470889943002053L, -4328870273523528678L)
    test(8363189190837666117L, 8952953554949964613L, -769908401025784465L)
    test(3498185872422470290L, 8258561189039573746L, 4075684841557452446L)
    test(152031744811368624L, 170047302979390384L, 188204032090751156L)
    test(-6947918360261991384L, -6920860997071969106L, -4638813752546857431L)
    test(4773888190228688898L, 4860586990709995570L, -2971977436771387582L)
    test(10278305733755026L, 4748065148087263387L, 161255014989813942L)
    test(-5727407743982063484L, -5715302060276321884L, -5056348510402596716L)
    test(-1287747740781772447L, -1261234657337996827L, -134615009666509469L)
    test(146419765681076362L, -5555066422590801761L, 5047336750771649418L)
    test(1161933938494163120L, 3868632503303470008L, -2871519996647158542L)
    test(2035794844651131392L, 2081113725736229380L, -6970203712651684119L)
    test(-8718956379931848639L, -8699816080424968763L, -1790009333525024183L)
    test(2305846170454917457L, -6592122016529403019L, 3699855408466037209L)
    test(1297085485057794242L, 6824147944731570370L, 1297719358344094450L)
    test(-2987000190520245183L, -653713302077782055L, -2410516622272607131L)
    test(-9074494313555000700L, -8497729801827538201L, -8757553481155291236L)
    test(-6340925317080538148L, -558285175668710436L, -5861774332979855362L)
    test(-2161164149023808496L, -960664228237319055L, -2143127760139164520L)
    test(320179444010717800L, 2698669699295908460L, -8757944750987462663L)
    test(144512116068450304L, -4088870903741410270L, 2500618527615928924L)
    test(-7888036226553954292L, -2623289758830370499L, -7870012890183970292L)
    test(36188363825353220L, 5255900585270516671L, 1333587896419014148L)
    test(6417771495000134724L, 9031034116186600774L, -2796520704900833579L)
    test(4616893315328639104L, -4243933831904942959L, 8698415720520667596L)
    test(576694201158470210L, 903521970964279106L, 2887041401113893459L)
    test(8567006515635945861L, 8643603040776227813L, -76737606277036649L)
    test(5916604216750661664L, -3215627100310413264L, 6249871729558647915L)
    test(2305856212826375944L, 3211134852027244428L, -6897740518823901333L)
    test(85570600736229700L, 2680018099529944532L, 1422164597223041380L)
    test(8079484120056629843L, 8105416119112362587L, 8476509488258980851L)
    test(1441758068147864099L, 1518812428010637107L, 8539457623816399467L)
    test(1894326628160946348L, 1895738751332569260L, -2356832824956688644L)
    test(2882919871120277520L, 4391625900913461531L, 2956119867420268048L)
  }

  @Test def bitwiseXor(): Unit = {
    @inline def test(expected: Long, x: Long, y: Long): Unit = {
      assertEquals(expected, x ^ y)
      assertEquals(expected, hideFromOptimizer(x) ^ y)
      assertEquals(expected, x ^ hideFromOptimizer(y))
      assertEquals(expected, hideFromOptimizer(x) ^ hideFromOptimizer(y))
    }

    test(-1345283394711251164L, -5494496810055058212L, 6839621630049102840L)
    test(1177739103851983454L, 6158022126886990489L, 4984810816587366599L)
    test(-2204300700029598283L, 1233870775205927883L, -1119330462532772226L)
    test(-7148123363625843479L, 7798122998092269952L, -1084034761905582743L)
    test(5963743598839966616L, 95326718809172319L, 6021825151394693831L)
    test(-6821612690598126759L, -362149276737089815L, 6606112428146152880L)
    test(648084059595035442L, -8458368415573046015L, -9051210810203578829L)
    test(150543288812241171L, -14578285135338579L, -154561155574275394L)
    test(2299862956880245212L, -7356743072970029538L, -8787270491260618814L)
    test(-6921750493429036678L, 8899282331008531693L, -1985635844405627497L)
    test(-7539137278621614021L, 7152725435954964647L, -856792829842315108L)
    test(3977512618797217577L, 7546071599906842763L, 6884876300521496482L)
    test(5787075022276678939L, -872709838480168559L, -6652853399531225974L)
    test(-1100292585378215294L, 6011211457303404974L, -6640868530527212756L)
    test(784693412557232428L, -7005516635337999530L, -7771896579505085830L)
    test(-536410976242375002L, 3875986038609442114L, -3655781831807888412L)
    test(7815158447936394893L, -88054661166499213L, -7876180077040489218L)
    test(2020578627487381781L, 2648592106093999733L, 4092428400936046432L)
    test(-6395160311373034714L, 4742133071334599144L, -1805763735648148786L)
    test(-441769954074007893L, -6657837103464807620L, 6504351236289652119L)
    test(6918489142066290938L, 5939064367543533499L, 3632331956590387009L)
    test(2433883990441869738L, 3127159199564273868L, 766458703092373862L)
    test(-8920139357303733331L, 4465379772372369781L, -5058250752714718504L)
    test(6491247119739672874L, -8054699132840626143L, -3881400233028001525L)
    test(7293310738954683159L, -1795834570369388381L, -9068874707512872012L)
    test(-1564358087740099369L, 8316440948344494714L, -7411897624314515795L)
    test(-6166268887740733107L, -5967425765583698090L, 523106950137081371L)
    test(474138983114521864L, -2352219494436833830L, -2751903668769699630L)
    test(-1796619272599084973L, 3697494534408094884L, -3152233891800308489L)
    test(6749975861603984239L, 9110049561117643145L, 2576585135545856742L)
    test(2918631955902790514L, -2782422835043071708L, -1016747674421759402L)
    test(-8180690364776263895L, -7191066992704591214L, 1318547397806302651L)
    test(4521254935253472413L, -3695203816927444106L, -1006195129194866709L)
    test(-7493949464502980537L, 4977644184000768067L, -2516384463912069116L)
    test(-7559040216863644878L, -3638074259892523847L, 6528550695483625355L)
    test(-6930185456427239817L, 7334461368042660015L, -424969169842121000L)
    test(-4761474216833548639L, -1695959000690469613L, 6169199399844162482L)
    test(-2384579340115197158L, -4499074431917642449L, 2267657096247078453L)
    test(-4934694780867541039L, 7843570073008701485L, -2928038698018994180L)
    test(5241140121429290918L, -942596195682218275L, -5019405910948372101L)
    test(-8761259447037421544L, 6389526908210287028L, -2394252346556151380L)
    test(2869237754534285562L, 8088172451025036055L, 6336390355576255469L)
    test(4815254228977302001L, 5599787830032599513L, 1109373601620275240L)
    test(3903284871130245430L, -4990754088192038910L, -8316451476840660684L)
    test(4907811280174376104L, -2044244740855885358L, -6359805712396061318L)
    test(-1191759457854524194L, 8312586765039472901L, -7193875294949415461L)
    test(1558396501139351174L, -3140618724783712169L, -4482543203453698351L)
    test(-8596716778517160802L, 1067526238207535946L, -8763214506171238444L)
    test(-3512891522172443941L, 1284007884260900389L, -2382960985657451266L)
    test(7352713776562101425L, 5100002577055830592L, 2363491965446098673L)
  }

  @Test def shiftLeft(): Unit = {
    @inline def test(expected: Long, x: Long, y: Int): Unit = {
      assertEquals(expected, x << y)
      assertEquals(expected, hideFromOptimizer(x) << y)
      assertEquals(expected, x << hideFromOptimizer(y))
      assertEquals(expected, hideFromOptimizer(x) << hideFromOptimizer(y))
    }

    test(-2970093265644158976L, 5448625119334748799L, -73329513)
    test(-4494976474830340096L, -1252320687782865211L, -339545061)
    test(0L, -7529362387021292180L, -809014658)
    test(-6356089297145167872L, 3641054366760334803L, 1030973528)
    test(7286824197085462528L, 3520413204675044946L, 82069876)
    test(-3278017996353699840L, -2558088160941399927L, 503039850)
    test(-5843420516513218560L, 4791828575890798045L, 1171866675)
    test(-3367538050735710208L, 1231323421663621707L, -1938771891)
    test(1888646915074293760L, -9165795994885001326L, 1537970769)
    test(4852401102343520L, 4035376903658412651L, 1856717061)
    test(5565618142278844416L, 2851140530886203800L, 1561097110)
    test(-4038616684974760040L, -7422356113262926861L, 858801923)
    test(-6541630578936633280L, -780886707895193278L, 1110910853)
    test(859616599361978368L, 3733313702748734961L, -338843811)
    test(-7453927974274859008L, -866510160693522860L, -813015128)
    test(9036894811516441600L, -441534870147678075L, -2122946486)
    test(3314649325744685056L, 5028237321003304750L, 2126487160)
    test(-3053018334892130304L, -7312385461262653344L, -32425558)
    test(-4611686018427387904L, 1756944750067922542L, 1222452733)
    test(-5080060379673919488L, 911926978090186099L, -834196361)
    test(6629298651489370112L, -7489247941380587858L, -1762017159)
    test(-288230376151711744L, 8331985217752613630L, 1042647417)
    test(5188146770730811392L, -2655334158172847278L, -753853766)
    test(-6954300141096140800L, 7504492034471689794L, 1382741393)
    test(-1286526581291876352L, 5767694384674461113L, 1929734882)
    test(-6809442636584189952L, -2132571568854329722L, -129147274)
    test(2708742116339613696L, 6353674613576604708L, 289081298)
    test(-2305843009213693952L, 4345082395622665255L, 370873533)
    test(438678751203557376L, -8715986565789905875L, -916638609)
    test(-4611686018427387904L, -6752063794606076436L, 1493241980)
    test(-7784308849075353252L, -3892154424537676626L, 2074806145)
    test(5885641763019816960L, -7602816679136650528L, -2062208020)
    test(7053568968128200704L, 1978935989718610781L, -902835237)
    test(7261635908661673984L, 3315013180757929043L, 868975579)
    test(2420500656866532864L, 4508327167715938447L, 575477257)
    test(-8228718371591885504L, 5250312624809742036L, 1238257604)
    test(-9223372036854775808L, -2335460374644938788L, 553354173)
    test(-8199764094790664192L, -2221296663047759285L, 1899246569)
    test(-2095845084401500160L, -3851886195238883679L, 1487779500)
    test(-3639161319019061248L, -802529200790849426L, -119255604)
    test(8334058599626047488L, 4498082071275250057L, 2089738146)
    test(9180988356411322368L, -4235645797120184753L, 800913356)
    test(7089417879434559488L, -1720882952904599381L, -1609938966)
    test(3856188238922252288L, 1093319069580253367L, -1518372133)
    test(-4742607066969931776L, 3772324202736502236L, -923142549)
    test(1312485114016636864L, -8626403704644842369L, -252023418)
    test(0L, -5643258165082180792L, 1174528765)
    test(-6881571249355748148L, -1720392812338937037L, -967606846)
    test(4994738277358501888L, -8565500659172726672L, 1237159401)
    test(7770778327110513462L, -5337982873299519077L, 484063041)
  }

  @Test def shiftLogicalRight(): Unit = {
    @inline def test(expected: Long, x: Long, y: Int): Unit = {
      assertEquals(expected, x >>> y)
      assertEquals(expected, hideFromOptimizer(x) >>> y)
      assertEquals(expected, x >>> hideFromOptimizer(y))
      assertEquals(expected, hideFromOptimizer(x) >>> hideFromOptimizer(y))
    }

    test(20858343375185L, 5467889565744671255L, 2099569298)
    test(40L, 5849044181494380798L, -2053535175)
    test(258L, 4664225633318643153L, -303705162)
    test(11295177044L, 6064052000889978176L, 1015183069)
    test(1802877774429066L, 923073420507681829L, 1191603401)
    test(110698435923972274L, -4277344275441100506L, 485744647)
    test(2268034701581L, -8933917052726000505L, -386336938)
    test(185L, -5090094480666808474L, 2048537528)
    test(45022L, -5773949593215980393L, -791372688)
    test(587L, -7863358856597183925L, -1259543946)
    test(124192589756539L, 4069542781142272099L, -1409717873)
    test(3L, -4039330657386967783L, -847604034)
    test(11L, 6646835208299080604L, -216455813)
    test(266L, -8837672319221710475L, 787633143)
    test(254846827215005840L, -2136547131949177847L, 1584315654)
    test(25694L, -3982242199062531977L, 1300572337)
    test(13872010164654615L, 1775617301075790739L, 1180537031)
    test(43758446423057L, 5735507089563036214L, -1668840943)
    test(42419379403L, -7059878620412332866L, -2118687716)
    test(26L, 3760648055340959723L, 350453433)
    test(245197086164441L, -2377507834836728200L, -1203541232)
    test(172610132266043L, -7134566445522111487L, 254766480)
    test(31181070059474408L, 3991176967612724340L, 1175504903)
    test(99018L, -4511082038238435159L, 629735727)
    test(16237L, 4570503775438788501L, 577897264)
    test(689994L, 6069259321955006666L, 243415787)
    test(4088L, -34322001027985900L, -1844808396)
    test(48441534L, -5131236380867176343L, -2102413146)
    test(42961906L, 5766249524366694L, 365466523)
    test(1946L, -917054041531076386L, -717261067)
    test(46638927393L, -5927202331435696117L, -675522340)
    test(8L, -8401337655395112344L, 192236860)
    test(1112244037586L, -9116564842061629491L, -1278414057)
    test(39284548082721288L, -8389899764532901884L, 2134440904)
    test(175996054L, -6352387266942387944L, -1873327132)
    test(771890457L, 1657622134816050076L, 1526047775)
    test(-5155643191806034313L, -5155643191806034313L, 93595840)
    test(88233L, -6029041726036498474L, 1625850351)
    test(21L, -6213512192619062443L, -742234373)
    test(200097858L, 6875310053473933879L, 1678034787)
    test(1L, 1914479947832224654L, 363036476)
    test(13467856720853L, 7061035664462656346L, -1297864237)
    test(27660L, 3892851735690760579L, -1204888593)
    test(563888993282225291L, 4511111946257802334L, -1598422013)
    test(61L, -830456149344766304L, 415626042)
    test(12362394L, -4854147173401088028L, -1259042456)
    test(36437577573404L, 2387973083850613402L, 1780437072)
    test(88L, 794207481683175979L, 1729784373)
    test(27109L, -3185446728240817200L, -204793551)
    test(47238811875L, 1585071500839360304L, 768150041)
  }

  @Test def shiftArithmeticRight(): Unit = {
    @inline def test(expected: Long, x: Long, y: Int): Unit = {
      assertEquals(expected, x >> y)
      assertEquals(expected, hideFromOptimizer(x) >> y)
      assertEquals(expected, x >> hideFromOptimizer(y))
      assertEquals(expected, hideFromOptimizer(x) >> hideFromOptimizer(y))
    }

    test(12083834796762671L, 3093461707971243921L, -1316031160)
    test(1519L, 3421928456457518817L, 1452874739)
    test(535935432373L, 4495752255492475708L, 1356453463)
    test(-15335L, -8632722372891668805L, 393061105)
    test(5L, 788322396658845547L, -1980770119)
    test(61456853258820810L, 7866477217129063791L, 1545082311)
    test(523169438L, 2246995629203373230L, -2119005984)
    test(-3140911288418704L, -3216293159340752798L, 511130378)
    test(-197222925856L, -3308851627222811261L, -1176556648)
    test(263770043357L, 1106331747933537319L, -1512809002)
    test(-51L, -7276147772649213050L, 2083093369)
    test(-5L, -2316677564967907265L, -207994821)
    test(-13942941898828745L, -7138786252200317415L, -638486135)
    test(-10706L, -3013225718780605504L, 611632432)
    test(431055124131099630L, 6896881986097594087L, 244039684)
    test(1L, 6268288981035101119L, -439796226)
    test(7L, 8640634673209662748L, -652735044)
    test(423071173443L, 3548978230116134282L, 613066583)
    test(-2989967169176L, -6270415628770328030L, -775565931)
    test(-3361820324L, -7219454171626000888L, -531660641)
    test(5575500511542107L, 1427328130954779450L, -434372344)
    test(-556401176231479L, -9116076871376536336L, -763866098)
    test(-72307L, -2544065484579384992L, -1841005139)
    test(72262L, 1271262762129413514L, 69079212)
    test(-102489556331053708L, -6559331605187437274L, 1371712838)
    test(70395261L, 4837525531312068659L, 1948323684)
    test(-329L, -5915257366775763309L, -1243200330)
    test(-88269909299L, -2961846669202422044L, 728732313)
    test(-62655L, -8817858119848136407L, -777879057)
    test(85473610109514853L, 85473610109514853L, 373478400)
    test(-1L, -7586855577422630242L, 942391743)
    test(5131L, 5777725881539893498L, -813574478)
    test(9L, 1351155571146391276L, -641829383)
    test(-14492L, -2039493177828382555L, -920501329)
    test(40L, 5807947005141331651L, -177471111)
    test(59705499154418508L, 7642303891765569108L, -508943033)
    test(-37529L, -5281723454849434235L, 401183471)
    test(376386L, 3310727509848511642L, 797274667)
    test(-1822L, -2050466318844521906L, 1298272370)
    test(-10994094310680L, -2882035858978840647L, 532386578)
    test(149987163970727L, 4914779388992785344L, -2131012273)
    test(-121723L, -8565430458495189455L, -1642432978)
    test(40689875474266L, 5333303358163034241L, 1681307793)
    test(-278047L, -4891437049391997940L, 2010442220)
    test(872834L, 1919382843392628399L, -1707024855)
    test(-1L, -4032486568803685L, -829231944)
    test(51533210804458L, 6754561006561936063L, 309495249)
    test(-11427L, -6432384051195600537L, -176819791)
    test(201L, 1812121086093687413L, 2056663541)
    test(-8694773540L, -2333985499857848703L, -617227620)
  }

  @Test def negate(): Unit = {
    @inline def test(expected: Long, x: Long): Unit = {
      assertEquals(expected, -x)
      assertEquals(expected, -hideFromOptimizer(x))
    }

    test(0L, 0L)
    test(1L, -1L)
    test(-1L, 1L)
    test(-9223372036854775807L, MaxVal)
    test(MinVal, MinVal)
    test(-4294967296L, 4294967296L)

    test(-6521757321054339836L, 6521757321054339836L)
    test(-5034215373457122166L, 5034215373457122166L)
    test(-1466708504867719503L, 1466708504867719503L)
    test(6932328637909659254L, -6932328637909659254L)
    test(-6036151501906311780L, 6036151501906311780L)
    test(-2049865194005705494L, 2049865194005705494L)
    test(-6415892737421190372L, 6415892737421190372L)
    test(7163344881632673520L, -7163344881632673520L)
    test(-5465469256934522955L, 5465469256934522955L)
    test(-5685869615186909998L, 5685869615186909998L)
    test(-2969720262713562574L, 2969720262713562574L)
    test(-49296126029544990L, 49296126029544990L)
    test(-8788528556936226133L, 8788528556936226133L)
    test(4038755528685525299L, -4038755528685525299L)
    test(-1074492420652895592L, 1074492420652895592L)
    test(-3622318736383541934L, 3622318736383541934L)
    test(4914594916508285209L, -4914594916508285209L)
    test(-7201284052855688325L, 7201284052855688325L)
    test(-3790499419214724571L, 3790499419214724571L)
    test(-8011289068563413537L, 8011289068563413537L)
    test(-3577068739332851405L, 3577068739332851405L)
    test(-6115071375844539143L, 6115071375844539143L)
    test(6526707827083299740L, -6526707827083299740L)
    test(3112673891190060801L, -3112673891190060801L)
    test(3003902505100987603L, -3003902505100987603L)
    test(2645609183394921729L, -2645609183394921729L)
    test(4998167914278755864L, -4998167914278755864L)
    test(5590940018975078416L, -5590940018975078416L)
    test(-4546350589039504044L, 4546350589039504044L)
    test(4931343496360764878L, -4931343496360764878L)
    test(52874837222654534L, -52874837222654534L)
    test(2228837764635947525L, -2228837764635947525L)
    test(-699644125147613407L, 699644125147613407L)
    test(-7560919991164957655L, 7560919991164957655L)
    test(-4740041656772685483L, 4740041656772685483L)
    test(6370025830897801731L, -6370025830897801731L)
    test(-5145272132530939636L, 5145272132530939636L)
    test(3801285492485871053L, -3801285492485871053L)
    test(-5580041728091891284L, 5580041728091891284L)
    test(1504295086290629779L, -1504295086290629779L)
    test(-1894169850815200418L, 1894169850815200418L)
    test(-5142345395752170872L, 5142345395752170872L)
    test(-2059614626708839072L, 2059614626708839072L)
    test(6764101357673133489L, -6764101357673133489L)
    test(1855883125141869234L, -1855883125141869234L)
    test(-8547851575036586975L, 8547851575036586975L)
    test(-2458791672488577437L, 2458791672488577437L)
    test(8600237054883635595L, -8600237054883635595L)
    test(3583990235354542721L, -3583990235354542721L)
    test(2957283069254138108L, -2957283069254138108L)
  }

  @Test def plus(): Unit = {
    @inline def test(expected: Long, x: Long, y: Long): Unit = {
      assertEquals(expected, x + y)
      assertEquals(expected, hideFromOptimizer(x) + y)
      assertEquals(expected, x + hideFromOptimizer(y))
      assertEquals(expected, hideFromOptimizer(x) + hideFromOptimizer(y))
    }

    test(-2433913414145288860L, -2433913415150419947L, 1005131087L)
    test(7190417094319483341L, 7190417094569730565L, -250247224L)
    test(-8440297071285254315L, -8440297070712201045L, -573053270L)
    test(-4182841114472128928L, -4182841115722360187L, 1250231259L)
    test(1930778912555952111L, 1930778910565887113L, 1990064998L)
    test(-3942694955800008028L, -3942694955083685134L, -716322894L)
    test(7705242192270010714L, 7705242190628372781L, 1641637933L)
    test(8088093439981558721L, 8088093441540626931L, -1559068210L)
    test(409778645801625887L, 409778644514166370L, 1287459517L)
    test(5014681192099551315L, 5014681192788233382L, -688682067L)
    test(-2608924423361531934L, -2608924424689477233L, 1327945299L)
    test(-8620418727428276566L, -8620418727962295010L, 534018444L)
    test(-5116811003765580457L, -5116811004266544790L, 500964333L)
    test(466797473625031396L, 466797473566381010L, 58650386L)
    test(548716640880004738L, 548716640744439601L, 135565137L)
    test(1844726773970894384L, 1844726772387382421L, 1583511963L)
    test(-2554638993633829919L, -2554638995607451402L, 1973621483L)
    test(3910455677776483847L, 3910455679107918564L, -1331434717L)
    test(1626611332802912373L, 1626611331062029479L, 1740882894L)
    test(-2935860100559672408L, -2935860098496441681L, -2063230727L)
    test(-7028619595191553451L, -7028619593075073585L, -2116479866L)
    test(-1453009017531579993L, -1453009018974127640L, 1442547647L)
    test(-2003489716391726903L, -2003489716706140086L, 314413183L)
    test(-6085991422431947023L, -6085991423171521406L, 739574383L)
    test(-8740933909842007164L, -8740933911862681503L, 2020674339L)
    test(5079670096634461164L, 5079670096843001759L, -208540595L)
    test(-8170676868318819829L, -8170676866768083707L, -1550736122L)
    test(-2606903430814531525L, -2606903432198282718L, 1383751193L)
    test(4566344032032694750L, 4566344030599324455L, 1433370295L)
    test(434403394228970688L, 434403392274991070L, 1953979618L)
    test(4488996435074282858L, 4488996436199997790L, -1125714932L)
    test(6469581209714515793L, 6469581207710964251L, 2003551542L)
    test(4756706154634446171L, 4756706156580331862L, -1945885691L)
    test(-247187529795235106L, -247187531335883991L, 1540648885L)
    test(-1782331786257542816L, -1782331784980649789L, -1276893027L)
    test(2604368800870123615L, 2604368802720259121L, -1850135506L)
    test(-2603928021483848477L, -2603928022120270391L, 636421914L)
    test(227415677200863830L, 227415678797843344L, -1596979514L)
    test(-2678141494828406821L, -2678141494185161961L, -643244860L)
    test(1595522451690042937L, 1595522451179947863L, 510095074L)
    test(-1080315956418065613L, -1080315958087557587L, 1669491974L)
    test(-1394000483448677229L, -1394000482879290678L, -569386551L)
    test(5393143255269010405L, 5393143254832815742L, 436194663L)
    test(3086080803617309222L, 3086080802694862574L, 922446648L)
    test(-3079483957727273141L, -3079483959413893522L, 1686620381L)
    test(-8959613381043138071L, -8959613379516582512L, -1526555559L)
    test(-5253410531799693622L, -5253410530151170203L, -1648523419L)
    test(-8308110282489704154L, -8308110283731993792L, 1242289638L)
    test(-322703747879389106L, -322703748856620959L, 977231853L)
    test(5852046712121097366L, 5852046713602404411L, -1481307045L)
  }

  @Test def minus(): Unit = {
    @inline def test(expected: Long, x: Long, y: Long): Unit = {
      assertEquals(expected, x - y)
      assertEquals(expected, hideFromOptimizer(x) - y)
      assertEquals(expected, x - hideFromOptimizer(y))
      assertEquals(expected, hideFromOptimizer(x) - hideFromOptimizer(y))
    }

    // Whitebox corner case
    test(-1L, 0L, 1L)

    test(1986061787443775719L, 1986061786531926741L, -911848978L)
    test(2002065766737908670L, 2002065767152142652L, 414233982L)
    test(-2462684939552419214L, -2462684939084788813L, 467630401L)
    test(-8508279235553297423L, -8508279237198417759L, -1645120336L)
    test(7839506195525549270L, 7839506197479642872L, 1954093602L)
    test(-4628932694616068680L, -4628932693148869261L, 1467199419L)
    test(-7840803876477289029L, -7840803875191655918L, 1285633111L)
    test(-8670153348175879923L, -8670153348443041492L, -267161569L)
    test(-1242495931707110914L, -1242495933164810772L, -1457699858L)
    test(-8507088039514506937L, -8507088038378074542L, 1136432395L)
    test(90780360945105021L, 90780359532567450L, -1412537571L)
    test(986525490439540924L, 986525489291348285L, -1148192639L)
    test(-8866371591861212506L, -8866371591422202167L, 439010339L)
    test(-8107568276862373051L, -8107568278647453278L, -1785080227L)
    test(7534812863279983696L, 7534812864750415746L, 1470432050L)
    test(-3631733870293241710L, -3631733869505495360L, 787746350L)
    test(-6352315350260551401L, -6352315348459271954L, 1801279447L)
    test(-9222685583547881455L, -9222685583355676355L, 192205100L)
    test(6893165777287389869L, 6893165778671654152L, 1384264283L)
    test(4374922365789659495L, 4374922367086388496L, 1296729001L)
    test(-668435350990710641L, -668435350030924400L, 959786241L)
    test(6592392433218350407L, 6592392432815573779L, -402776628L)
    test(-3087465002163696041L, -3087465003742869350L, -1579173309L)
    test(7697932253104728967L, 7697932254039254286L, 934525319L)
    test(-4198280913984699596L, -4198280914515840124L, -531140528L)
    test(-6584160250140365720L, -6584160251001882531L, -861516811L)
    test(-4066515537811646002L, -4066515538088171194L, -276525192L)
    test(-3582933490129177640L, -3582933488925390211L, 1203787429L)
    test(-3997034206135429889L, -3997034205021042356L, 1114387533L)
    test(-1816067228069450094L, -1816067230174222304L, -2104772210L)
    test(7901705968383945410L, 7901705967077018212L, -1306927198L)
    test(-7098896964384888299L, -7098896965558095200L, -1173206901L)
    test(7691595054538136496L, 7691595056391915873L, 1853779377L)
    test(1185121830164420906L, 1185121831425231077L, 1260810171L)
    test(2828560744907836028L, 2828560746155873666L, 1248037638L)
    test(-8972435051651997701L, -8972435052203182058L, -551184357L)
    test(4262481960874455993L, 4262481961621435585L, 746979592L)
    test(-5121922580566520947L, -5121922580844970570L, -278449623L)
    test(-2934566834100100266L, -2934566833279842160L, 820258106L)
    test(2069114446928198006L, 2069114445924505589L, -1003692417L)
    test(6064340163069368882L, 6064340162935355320L, -134013562L)
    test(-933615963411001916L, -933615964053308835L, -642306919L)
    test(1195021009572561699L, 1195021010578791501L, 1006229802L)
    test(8824806479059281365L, 8824806478386318374L, -672962991L)
    test(1124449446003809225L, 1124449446837255411L, 833446186L)
    test(6986942923034886189L, 6986942924704869081L, 1669982892L)
    test(-7361868769204923739L, -7361868771289309455L, -2084385716L)
    test(-2335066428381605387L, -2335066428056987903L, 324617484L)
    test(-2587674234131885903L, -2587674234309061950L, -177176047L)
    test(4156626309610486835L, 4156626310084674081L, 474187246L)
  }

  @Test def times(): Unit = {
    @inline def test(expected: Long, x: Long, y: Long): Unit = {
      assertEquals(expected, x * y)
      assertEquals(expected, hideFromOptimizer(x) * y)
      assertEquals(expected, x * hideFromOptimizer(y))
      assertEquals(expected, hideFromOptimizer(x) * hideFromOptimizer(y))
    }

    test(4466388609482222752L, 5036730703751139101L, 1721031968L)
    test(-5306406633626451618L, 6841001363030916138L, -850659149L)
    test(-2089429074589014090L, 6479175383404433070L, -2054981347L)
    test(8460323636954078769L, 4758350191766752285L, 324089381L)
    test(-5923156225012284416L, -9220199974010895616L, 2043533548L)
    test(2209979247360766336L, 402330037909496490L, 1272468928L)
    test(-740541357758937180L, 4408402409782238012L, 244738503L)
    test(2871901923443243387L, 1825457782616022753L, 750432219L)
    test(9125337863049267666L, 8778397201268811722L, -1367964491L)
    test(3711682045754580479L, -4708639793668596655L, -461844145L)
    test(1732264674836666559L, -1424969076694017025L, 1495188289L)
    test(-8346467091089544904L, -1559241233428019412L, -1134101206L)
    test(-8095884306901452544L, -1856101761515477674L, -488368768L)
    test(-9042873153403997255L, 5072892489253867083L, -1852476533L)
    test(4053715608733663552L, -6534575507831614492L, -235907376L)
    test(9220795863880349128L, -6669028838631715807L, 990887112L)
    test(-7439311185174790639L, 1658475494989623827L, 1634792395L)
    test(3682932726430389338L, -2535802754766948607L, 799604314L)
    test(-4254078436736665200L, -5278482193701750342L, 1440046952L)
    test(-4111927075334805665L, 7555445675614168645L, -156634285L)
    test(2557607844416221682L, 2257106508626305843L, 600761926L)
    test(629063292600453823L, 3408004029612789043L, 1986597957L)
    test(-6281137167850793591L, 3565530158654203649L, -1711884663L)
    test(-8960699083603279762L, 7741229999055406521L, 1763957470L)
    test(-4659748226969785840L, 586638899445089560L, -241565738L)
    test(6421423204109643806L, 5432696032973354067L, 1823828906L)
    test(-6327137676645557459L, -9041036347144385291L, 48280153L)
    test(-2450619808166955997L, -6984024658713508935L, 301685947L)
    test(6215624234851202252L, -2968317789301520388L, -889918259L)
    test(-5714680300641087143L, -6052471925706112475L, -996910555L)
    test(7626829871011888526L, -5598604266227803174L, -1761056509L)
    test(-336850264806741348L, -5131017450105726124L, -1549721205L)
    test(6070932791384448829L, 1417786036454722783L, 1414822755L)
    test(7948063376841928368L, 6196657643428237716L, -948878276L)
    test(-5967293703566631097L, 3321209290390227081L, 1294488911L)
    test(3523504090718067750L, 3447048083977045358L, 1394107269L)
    test(5048267536820983632L, 1021477279097134774L, -763790472L)
    test(4190715788841493798L, -1591527307038089513L, 1449901386L)
    test(-1622739101396600271L, 1584681944074230843L, 1406834819L)
    test(-1343991317807294247L, -3430617443510905869L, -1861488893L)
    test(-3418544184172942166L, 8236081087714485202L, 2083344781L)
    test(1667294848924673254L, -1899145842670901051L, -552836178L)
    test(-679895375193063706L, 886032925265918614L, -1547545223L)
    test(3479342126707702906L, 7265546667600848990L, 1172634963L)
    test(8579781526146211328L, 214039809610896270L, -1645815552L)
    test(6655434057463364804L, -8344930817859169419L, -295254732L)
    test(-1618504871616397322L, -8524408781207533066L, -1583554303L)
    test(-719194663978943528L, -8840851638735986326L, 1620104636L)
    test(-134629840588182168L, -3318104751418409923L, 21655944L)
    test(-5452818611351983605L, -459718335291242471L, 2077087683L)

    test(8433193943336928478L, -304510477859059605L, -504694402761190L)
    test(-12731773183499098L, -253162060478L, 50291L)
    test(0L, 0L, -13850059L)
    test(7569251612557229982L, -8660470952582643L, -874L)
    test(-11988L, -1332L, 9L)
    test(-8243580206627053600L, -29568141078178L, 1526544L)
    test(2184313243348463060L, 328926390054L, -3638668370L)
    test(46841680L, 205L, 228496L)
    test(740594256954004614L, -19460467868573L, -81306407837343422L)
    test(1686350941924289808L, 515501176792L, -14676235751610L)
    test(6344118389112076765L, 414866483823975467L, 470120246452409879L)
    test(-11349L, -117L, 97L)
    test(-15402750L, 30L, -513425L)
    test(-1358824991029065112L, 1375729456898L, 623181770548L)
    test(15724552950015L, 80638733077L, 195L)
    test(53496875011846994L, -12740384618206L, -4199L)
    test(4976657084923555180L, 521743856055513645L, -2105465236503908L)
    test(8L, -8L, -1L)
    test(619912072L, -34L, -18232708L)
    test(-4326473264912647477L, -4512416881041611L, 16599101951L)
    test(1570555712220296245L, -2433585405235L, -645367L)
    test(17255933394228520L, -774590L, -22277506028L)
    test(5693979142683511208L, 36307900L, -19149614702538L)
    test(5058564788733665886L, 22169162314093L, 18534166L)
    test(-1912529786602316571L, -3848931303L, 82157326906201261L)
    test(-5528746562555987920L, 47326191440L, -39094697833L)
    test(-2254385599727553792L, -14641542L, -3714491797523081344L)
    test(7866519297L, 771303L, 10199L)
    test(0L, 0L, -14806105L)
    test(47573376L, 3964448L, 12L)
    test(-2609453654630496L, 77505454872L, -33668L)
    test(-3686637842539497440L, 14929228532112L, 6555913938L)
    test(4853210716974444062L, 431219964946864070L, 1181669L)
    test(-14100245424035L, 2820049084807L, -5L)
    test(8725676311339308590L, -16830140929953126L, 93975690486771L)
    test(2367313232506909772L, 206188838L, 1249841574949634L)
    test(-8124607316971866814L, 16674057030L, 1104000290638571L)
    test(6446979988520042261L, -64710169253973867L, -23079009995647L)
    test(654519384576L, 31096512L, 21048L)
    test(153203910181224144L, 22L, 6963814099146552L)
    test(1436126772314869678L, 1260318190682L, 1123567398313107L)
    test(-75661570L, -5L, 15132314L)
    test(6685324216344409292L, 9099845427374L, -126975734L)
    test(-7100290L, 5L, -1420058L)
    test(-1940696769116022576L, -28274L, 68638918056024L)
    test(-7932625013377175292L, 193355246244L, 36593871833L)
    test(-7L, 1L, -7L)
    test(-48355929429178192L, 66789957775108L, -724L)
    test(-10521672279471L, 10521672279471L, -1L)
    test(4095350372293300139L, -8747667231979L, -105906241L)

    // Random power of 2 tests
    test(105129441230848L, 100259248L, 1048576L)
    test(105129441230848L, 1048576L, 100259248L)
    test(1297036692682702848L, 72L, 18014398509481984L)
    test(1297036692682702848L, 18014398509481984L, 72L)
    test(-11947131749269504L, -22253267L, 536870912L)
    test(-11947131749269504L, 536870912L, -22253267L)
    test(8659858730206101504L, 23022568162358L, 8388608L)
    test(8659858730206101504L, 8388608L, 23022568162358L)
    test(207805415948288L, 1548271L, 134217728L)
    test(207805415948288L, 134217728L, 1548271L)
    test(0L, -55880L, -9223372036854775808L)
    test(0L, -9223372036854775808L, -55880L)
    test(-2199023255552L, -1L, 2199023255552L)
    test(-2199023255552L, 2199023255552L, -1L)
    test(851968L, 13L, 65536L)
    test(851968L, 65536L, 13L)
    test(-17592186044416L, -1L, 17592186044416L)
    test(-17592186044416L, 17592186044416L, -1L)
    test(-7000097952840548352L, 222527207082L, 4398046511104L)
    test(-7000097952840548352L, 4398046511104L, 222527207082L)
    test(2449958197289549824L, 34L, 72057594037927936L)
    test(2449958197289549824L, 72057594037927936L, 34L)
    test(-16623318531928064L, -4058427375959L, 4096L)
    test(-16623318531928064L, 4096L, -4058427375959L)
    test(-8006274237557899264L, -2214324316485807900L, 281474976710656L)
    test(-8006274237557899264L, 281474976710656L, -2214324316485807900L)
    test(-61568356188160L, -14335L, 4294967296L)
    test(-61568356188160L, 4294967296L, -14335L)
    test(-3101185093760L, -48456017090L, 64L)
    test(-3101185093760L, 64L, -48456017090L)
    test(-20250091264L, -158203838L, 128L)
    test(-20250091264L, 128L, -158203838L)
    test(-118778880L, -115995L, 1024L)
    test(-118778880L, 1024L, -115995L)
    test(-4052041596928L, -483041L, 8388608L)
    test(-4052041596928L, 8388608L, -483041L)
    test(2511601217189183488L, 186114971352L, 35184372088832L)
    test(2511601217189183488L, 35184372088832L, 186114971352L)
    test(258064524401082304L, 8064516387533822L, 32L)
    test(258064524401082304L, 32L, 8064516387533822L)
    test(-13379618472345600L, -816627104025L, 16384L)
    test(-13379618472345600L, 16384L, -816627104025L)
    test(-56908316672L, -106L, 536870912L)
    test(-56908316672L, 536870912L, -106L)
    test(0L, 0L, 4096L)
    test(0L, 4096L, 0L)
    test(-513636086841344L, -61230193L, 8388608L)
    test(-513636086841344L, 8388608L, -61230193L)
    test(-3894698884001169408L, 41500264L, 8796093022208L)
    test(-3894698884001169408L, 8796093022208L, 41500264L)
    test(1055744L, 32992L, 32L)
    test(1055744L, 32L, 32992L)
    test(558955633836032L, 65071L, 8589934592L)
    test(558955633836032L, 8589934592L, 65071L)
    test(-7994738298998226944L, 76048351L, 137438953472L)
    test(-7994738298998226944L, 137438953472L, 76048351L)
    test(-4L, -2L, 2L)
    test(-4L, 2L, -2L)
    test(-4259162954240L, -8318677645L, 512L)
    test(-4259162954240L, 512L, -8318677645L)
    test(2922836158163451904L, -60717806L, 2251799813685248L)
    test(2922836158163451904L, 2251799813685248L, -60717806L)
    test(0L, 0L, 8192L)
    test(0L, 8192L, 0L)
    test(5531246788608L, 164844L, 33554432L)
    test(5531246788608L, 33554432L, 164844L)
    test(8574853690513424384L, -678234761L, 72057594037927936L)
    test(8574853690513424384L, 72057594037927936L, -678234761L)
    test(-2828260565988671488L, -103657850088L, 2251799813685248L)
    test(-2828260565988671488L, 2251799813685248L, -103657850088L)
    test(-9223372036854775808L, 138748537820112453L, -9223372036854775808L)
    test(-9223372036854775808L, -9223372036854775808L, 138748537820112453L)
    test(0L, 0L, 17179869184L)
    test(0L, 17179869184L, 0L)
    test(5526109039206858752L, -2489041709915087415L, 2147483648L)
    test(5526109039206858752L, 2147483648L, -2489041709915087415L)
    test(9663676416L, 9L, 1073741824L)
    test(9663676416L, 1073741824L, 9L)
    test(648518346341351424L, 379085341609132041L, 72057594037927936L)
    test(648518346341351424L, 72057594037927936L, 379085341609132041L)
    test(-4118880446592909312L, 218625848802439L, 65536L)
    test(-4118880446592909312L, 65536L, 218625848802439L)
    test(-7094878839486021632L, -422887732952L, 16777216L)
    test(-7094878839486021632L, 16777216L, -422887732952L)
    test(20894747459584L, 77839L, 268435456L)
    test(20894747459584L, 268435456L, 77839L)
    test(-5624995934585749504L, -3953804003778L, 2251799813685248L)
    test(-5624995934585749504L, 2251799813685248L, -3953804003778L)
    test(76001593786368L, 141564L, 536870912L)
    test(76001593786368L, 536870912L, 141564L)
    test(5942088864628736L, 11068003L, 536870912L)
    test(5942088864628736L, 536870912L, 11068003L)
    test(-417996242432L, -816398911L, 512L)
    test(-417996242432L, 512L, -816398911L)
    test(10122752L, 79084L, 128L)
    test(10122752L, 128L, 79084L)
    test(712964571136L, 166L, 4294967296L)
    test(712964571136L, 4294967296L, 166L)
    test(-3013501831155286016L, -120218862620908531L, 4294967296L)
    test(-3013501831155286016L, 4294967296L, -120218862620908531L)
  }

  @Test def divide(): Unit = {
    @inline def test(expected: Long, x: Long, y: Long): Unit = {
      assertEquals(expected, x / y)
      assertEquals(expected, hideFromOptimizer(x) / y)
      assertEquals(expected, x / hideFromOptimizer(y))
      assertEquals(expected, hideFromOptimizer(x) / hideFromOptimizer(y))
    }

    test(IntMaxValPlus1, IntMinVal, -1L)
    test(-1L, IntMinVal, IntMaxValPlus1)
    test(IntMinVal, IntMaxValPlus1, -1L)
    test(-1L, IntMaxValPlus1, IntMinVal)

    test(-9223372036854775807L, MaxVal, -1L)
    test(MinVal, MinVal, 1L)
    test(MinVal, MinVal, -1L)

    // int32, int32
    test(1L, -10426835L, -6243356L)
    test(-291L, 49659080L, -170373L)
    test(3L, 97420L, 27521L)
    test(26998L, -9881291L, -366L)
    test(0L, -40L, 81L)
    test(0L, -6007L, -326806L)
    test(-1L, 202L, -112L)
    test(0L, 0L, 47L)
    test(323816L, 22667160L, 70L)
    test(0L, 254L, -307349204L)
    test(0L, -17L, -44648L)
    test(-40L, 39646L, -976L)
    test(0L, 9L, 315779722L)
    test(0L, -2674L, -3051991L)
    test(0L, -37697L, 2015928L)
    test(0L, -13L, -31L)
    test(0L, 6L, -334L)
    test(8L, -15989L, -1918L)
    test(8746L, -113261535L, -12950L)
    test(55322L, -6362112L, -115L)
    test(0L, 455L, 13919L)
    test(36190L, 293468259L, 8109L)
    test(1L, -48287007L, -27531186L)
    test(349634L, 1048904L, 3L)
    test(0L, -34L, 3949717L)
    test(-1L, 1449L, -983L)
    test(-18537151L, 18537151L, -1L)
    test(0L, 14037L, 23645L)
    test(-4L, 1785L, -398L)
    test(0L, 346L, 2198158L)
    test(-802L, -3517419L, 4381L)
    test(-6L, 6L, -1L)
    test(39L, -822L, -21L)
    test(0L, 3629L, 282734L)
    test(-92367L, -278856469L, 3019L)
    test(0L, -13L, 37L)
    test(0L, -4L, 47150459L)
    test(0L, -26L, -210691L)
    test(0L, -21294L, 156839456L)
    test(0L, -5L, -25644L)
    test(0L, -1009L, 28100L)
    test(-857L, 16282815L, -18989L)
    test(-7L, -2201086L, 276963L)
    test(-300L, 11412578L, -37989L)
    test(0L, 8406900L, 239727371L)
    test(0L, -1L, -479069L)
    test(0L, 4L, -21776L)
    test(-16812960L, -16812960L, 1L)
    test(0L, 10873L, 57145L)
    test(0L, -1L, -7L)

    // int32, int53
    test(0L, -6975858L, 60171769780L)
    test(0L, -1L, 355344489879452L)
    test(0L, 43L, 213146146419L)
    test(0L, 4063968L, 102264144491298L)
    test(0L, -388987094L, 7404281630149L)
    test(0L, 5939808L, 53952090167L)
    test(0L, 7L, 5767755469224L)
    test(0L, -1175803932L, 10342578897871L)
    test(0L, 464610492L, 11927954100510L)
    test(0L, 214483L, 34696248634358L)

    // int32, big
    test(0L, 211494165L, 63586944937958278L)
    test(0L, 5353L, -1694817224433481042L)
    test(0L, 2926L, -974161160379320633L)
    test(0L, -6L, -2493566152636302714L)
    test(0L, -8L, 21133884687603971L)
    test(0L, -585878041L, -5566768138983983837L)
    test(0L, -4L, 275171713246363278L)
    test(0L, 34L, -80110932732961966L)
    test(0L, 785315L, 127658607682996801L)
    test(0L, 713146L, 11714632116570671L)

    // int53, int32
    test(-578207L, 228797600580841L, -395701427L)
    test(3734905142L, 11204715428L, 3L)
    test(80677703530L, 242033110590L, 3L)
    test(-162400270L, 4957105857076L, -30524L)
    test(-9L, 6323344774L, -691707459L)
    test(135006L, 314919788164L, 2332622L)
    test(-51739881980L, 6001826309794L, -116L)
    test(1125157L, 2639620573L, 2346L)
    test(997096L, 24421382294514L, 24492497L)
    test(-1295710758066L, 47941298048449L, -37L)
    test(-2166511L, 15114349972511L, -6976354L)
    test(-6909144294L, 207274328847L, -30L)
    test(4387003627L, 4101848391931L, 935L)
    test(23215066L, 4566357066146L, 196698L)
    test(-13221428L, 91740281077293L, -6938757L)
    test(-8625267371963L, 2785961361144320L, -323L)
    test(7125522617558L, 2087778126944550L, 293L)
    test(3373132868549L, 3305670211178975L, 980L)
    test(-917576L, 12579050561L, -13709L)
    test(456092L, 73591818663L, 161353L)
    test(1615312694926L, 3230625389852L, 2L)
    test(167998806719L, 167998806719L, 1L)
    test(90893135L, 132840499118665L, 1461502L)
    test(799104733L, 147585054932536L, 184688L)
    test(-300693860803928L, 601387721607857L, -2L)
    test(6770260858L, 1835756231761675L, 271150L)
    test(-21594175253L, 152282123889326L, -7052L)
    test(32825L, 3167385820L, 96492L)
    test(-57018115L, 31110452512408L, -545624L)
    test(-5950946L, 9505144073367L, -1597249L)
    test(-18245832494939L, 54737497484818L, -3L)
    test(4741011L, 27865199652383L, 5877480L)
    test(196504301807L, 564753363395315L, 2874L)
    test(228327608L, 4794879780L, 21L)
    test(99111506L, 36368273626634L, 366943L)
    test(-1209485521L, 24189710420L, -20L)
    test(-319956618L, 236542327972199L, -739295L)
    test(-62197L, 187572612454L, -3015755L)
    test(9087L, 246218266185L, 27093454L)
    test(345582531L, 8543663105119304L, 24722497L)
    test(-1424974L, 3807993484425L, -2672324L)
    test(1991351L, 153342025764L, 77004L)
    test(1193137L, 3507787521536L, 2939970L)
    test(573585390L, 531695875580133L, 926969L)
    test(-4044176128928L, 986778975458576L, -244L)
    test(-6019138L, 31632046988853L, -5255245L)
    test(124430635410L, 82124219370732L, 660L)
    test(-40732946L, 34018853638873L, -835168L)
    test(1280616075818L, 2561232151636L, 2L)
    test(-570078780L, 36329410438127L, -63727L)

    // int53, int53
    test(1L, 124003063371212L, 95971058218833L)
    test(0L, 52685446825193L, 65719008167515L)
    test(32L, 984680925730L, 30321302738L)
    test(368L, 16780572018393L, 45587849994L)
    test(1L, 224456523282412L, 136858431513934L)
    test(0L, 6270846242295L, 29713451361156L)
    test(0L, 5162639886L, 4241682616681008L)
    test(0L, 11906057278L, 898368875387L)
    test(0L, 8292364263L, 840690475571994L)
    test(0L, 24734473349L, 302851164091L)
    test(0L, 65461652427L, 132370406563769L)
    test(0L, 65169061341L, 6819803382011L)
    test(67766L, 999543310705255L, 14749729876L)
    test(694L, 677827034369276L, 975474741618L)
    test(1L, 474647334008012L, 411440625113020L)
    test(0L, 2139840204062L, 4997452565907L)
    test(12L, 147843129520789L, 12245680216440L)
    test(0L, 363347083296L, 63223041418797L)
    test(1L, 10444738372L, 8287074475L)
    test(131L, 459942257253L, 3488195032L)
    test(0L, 4872361206145L, 145063212196921L)
    test(0L, 423685951943L, 852441814396L)
    test(315L, 136483526964813L, 432278190260L)
    test(0L, 78818316243L, 4193722747205L)
    test(1L, 16202047394459L, 15528748620191L)
    test(2340149L, 5136010432017865L, 2194735964L)
    test(0L, 78321121408L, 248491421719L)
    test(14L, 1532585569763139L, 108379170191825L)
    test(131L, 2195472500094L, 16704059603L)
    test(0L, 288897937097L, 4174872075961L)
    test(1L, 21991187412389L, 15489126165811L)
    test(0L, 5839012516L, 10106143423983L)
    test(8L, 229000703832L, 27780654407L)
    test(0L, 58730811340L, 225578259160L)
    test(0L, 57432730865L, 690482805325L)
    test(0L, 176831982665L, 44502943027168L)
    test(0L, 135650155666L, 2473601103342862L)
    test(76L, 4306939188043L, 56039151891L)
    test(0L, 27678403241L, 14530933230495L)
    test(0L, 325783201862L, 1429918316597450L)
    test(0L, 638270432082L, 3919909688035L)
    test(0L, 305398988952L, 6832490296772512L)
    test(0L, 2570041898L, 167230554267L)
    test(0L, 68712734660L, 751812070666L)
    test(50L, 8994542324058741L, 176704424639976L)
    test(0L, 3296554204L, 127090317995L)
    test(0L, 14870954195L, 204066183786L)
    test(0L, 8480319499L, 85970293968645L)
    test(127L, 3872913127538L, 30378210782L)
    test(0L, 285099136391851L, 1075947159560094L)

    // int53, big
    test(0L, 59305199701875L, -61978378594400103L)
    test(0L, 878702206301718L, -3047361456851090070L)
    test(0L, 31936398707065L, 679920552742984365L)
    test(0L, 1065886403865357L, 278654289724641174L)
    test(0L, 2667310741936L, 440847768145795128L)
    test(0L, 12278863038908L, -71793457772450809L)
    test(0L, 11618498453L, 110861759245861602L)
    test(0L, 95042014006L, -142554989424244815L)
    test(0L, 496460930879730L, -9178143056234401L)
    test(0L, 347683226692935L, 6318124517041566307L)

    // big, int32
    test(26192471907L, 723782312359256698L, 27633219L)
    test(879107723762463L, -2616224585917092202L, -2976L)
    test(189814434L, -64599774627614709L, -340331202L)
    test(-4860916252638991L, -505535290274455145L, 104L)
    test(61885399461685L, -43752977419411966L, -707L)
    test(8600250819417791L, 722421068831094472L, 84L)
    test(-5157240553L, 156686648156150146L, -30381877L)
    test(-301408991367L, 2164461671311657366L, -7181145L)
    test(-10897544612214L, -37825377348996138L, 3471L)
    test(20273603415L, 41538261659499635L, 2048884L)
    test(1324107666L, 17884174072385720L, 13506586L)
    test(27320767651826108L, 54641535303652216L, 2L)
    test(-327955404958810L, 79037252595073238L, -241L)
    test(-177595108878660314L, 887975544393301570L, -5L)
    test(-76516770852181L, -260845671835087356L, 3409L)
    test(-2922506998444L, 10792818345254164L, -3693L)
    test(-148773795060L, -41563232673777087L, 279372L)
    test(2588091355L, -9757298518375933L, -3770075L)
    test(46855338940916L, -27878926669845136L, -595L)
    test(-1534866990081L, -11101692939261578L, 7233L)
    test(-1607988232266227L, 527420140183322469L, -328L)
    test(1421525100L, -68713435799727319L, -48337828L)
    test(-9422344923772683L, -141335173856590245L, 15L)
    test(-264609900256179L, 510697107494427095L, -1930L)
    test(-67052898569093L, -9454458698242151L, 141L)
    test(10084472962553L, -1916019609466306523L, -189997L)
    test(526621590143L, 231952059243267540L, 440453L)
    test(81459474793L, 234414545801980405L, 2877683L)
    test(-102870074701433L, 143298014059096433L, -1393L)
    test(-178073210L, -10447052378290802L, 58667176L)
    test(611202294492L, -233502502183290181L, -382038L)
    test(75406112292668151L, 226218336878004455L, 3L)
    test(2169515162020208600L, -4339030324040417200L, -2L)
    test(9939764678148L, -93950655737855096L, -9452L)
    test(3705731170438873170L, -7411462340877746340L, -2L)
    test(17364248365L, 6513292159132795798L, 375097846L)
    test(1243945230L, -1015249224614937988L, -816152673L)
    test(-3759636477941L, 113875629280371731L, -30289L)
    test(2336087242176329L, -3606918701920252251L, -1544L)
    test(102272964513L, -238477950920641560L, -2331779L)
    test(-10696841840L, -7709668252031154692L, 720742474L)
    test(-531682844470L, 279278111080786291L, -525272L)
    test(5393369008070L, -53281092430729833L, -9879L)
    test(3117012283L, 482514502098834968L, 154800321L)
    test(-219145567178234740L, -438291134356469480L, 2L)
    test(-1331247316563774L, 34612430230658145L, -26L)
    test(5269935045196L, -4126327520778768141L, -782994L)
    test(24765938495865L, -469636491697100502L, -18963L)
    test(181706959279836402L, 6178036615514437691L, 34L)
    test(-1886837760507410L, -26415728647103752L, 14L)

    // big, int53
    test(88399L, 1547294638316862730L, 17503470499928L)
    test(-45459L, -209839210582638165L, 4616002148581L)
    test(4395497L, 940370097701555719L, 213939423948L)
    test(3198134L, 2578616978236540319L, 806287894342L)
    test(470L, 3633455166312612934L, 7724173057120691L)
    test(92L, 27853465488661812L, 301778663579871L)
    test(167728L, 1806252011061650438L, 10768880197972L)
    test(25700177L, 2242463707657038031L, 87254795475L)
    test(-35822646L, -562598053333099173L, 15705094769L)
    test(-348L, -9112381674112046L, 26143402564658L)
    test(-2179L, -509057682015062058L, 233613517042906L)
    test(1187L, 181474883994203019L, 152819267878488L)
    test(3223L, 150553666516195580L, 46704391768104L)
    test(23608500L, 300363587202603124L, 12722688144L)
    test(7286803L, 45378843712712738L, 6227538127L)
    test(-137450L, -71343536160388549L, 519050448589L)
    test(114592L, 75617491646984375L, 659882766920L)
    test(61L, 59861182577767050L, 977967428486591L)
    test(-247360L, -3675261108362052533L, 14857923710053L)
    test(-61725L, -17706760935497774L, 286863193667L)
    test(2226L, 785774864008795364L, 352922104422398L)
    test(-997L, -33536475745382905L, 33616480572407L)
    test(36L, 19186635942791762L, 519428459800194L)
    test(56005L, 2626558083934978484L, 46898367867780L)
    test(-90057L, -79469909799080191L, 882439775947L)
    test(-9L, -14634507588329350L, 1515865418406015L)
    test(62240L, 2074357853073880797L, 33328209754855L)
    test(-1774L, -18883471414475441L, 10639493264514L)
    test(-153400L, -11246877030282940L, 73317116228L)
    test(2145L, 274737714555168746L, 128035347707168L)
    test(106042L, 185872670829287234L, 1752819113274L)
    test(-381407L, -167197834434549929L, 438370276333L)
    test(1217514L, 159606413062972309L, 131091960664L)
    test(-17L, -46712745604586317L, 2730376203303889L)
    test(4880327L, 8239833148507419484L, 1688377060863L)
    test(-1023070L, -2195881405872704768L, 2146364670600L)
    test(439L, 194204736450122813L, 442052361992590L)
    test(2171202L, 63817733899474393L, 29392809781L)
    test(-5332527L, -2641202944529830332L, 495300414087L)
    test(-435180L, -2269494636969765724L, 5215063874376L)
    test(27837L, 2314753811624982214L, 83152993851334L)
    test(-396L, -1594166273043442864L, 4023963601415498L)
    test(-115583279L, -892027830791363054L, 7717620044L)
    test(-6783543L, -450355256733558628L, 66389385105L)
    test(-1464006069L, -5807365214353469879L, 3966763072L)
    test(11599107L, 141701763851325632L, 12216609367L)
    test(842L, 1339898823144275017L, 1589772675238524L)
    test(43514L, 1753445026367710966L, 40295956575370L)
    test(-230L, -17726634007299531L, 77015681672071L)
    test(249094L, 276740732551840706L, 1110987688070L)

    // big, big
    test(-10L, -3035292483719699986L, 278502589500941238L)
    test(0L, 309352054257937604L, 3761260870078728814L)
    test(-2L, 36003686779005089L, -12888055118407988L)
    test(0L, 100264685753707774L, -1264190786717094832L)
    test(-37L, 461625058320865256L, -12195431348931304L)
    test(3L, 130723244245954867L, 35331266536894367L)
    test(-3L, -924932423900552316L, 252217550766181432L)
    test(2L, 69196492400756053L, 23535447678190438L)
    test(0L, 31449474321301746L, 9197997245108567284L)
    test(5L, 3736171329516923512L, 727045796882164310L)
    test(0L, 425057195577712622L, 2272184365474796943L)
    test(-2L, 807082311223102834L, -351978682494614360L)
    test(0L, -18868872832156821L, 226172537255253600L)
    test(22L, 572602005562282640L, 25982734904706641L)
    test(0L, 57843482432513353L, -2888397807038685054L)
    test(0L, -728953506047215037L, 3310993973484462547L)
    test(-27L, -1297762999973373449L, 47730487036641355L)
    test(-6L, -630351948303467419L, 101862614519465124L)
    test(0L, -161760942000700532L, 896669025309325774L)
    test(0L, -60571160055601992L, 3199697903948807752L)
    test(176L, -8905725250420237724L, -50384839403275195L)
    test(0L, 83529788996273036L, -868755088371510182L)
    test(6L, -227446351753346408L, -33328529234649391L)
    test(1L, 8187661920961984367L, 5363684730160280818L)
    test(0L, 189508821671714868L, -2046527394535651556L)
    test(0L, 156063739985865739L, -2214871552090474931L)
    test(0L, -71987272961247290L, -188911916350736042L)
    test(0L, 110913222542181819L, 1237912121737071001L)
    test(0L, 12880539371492270L, -9119362639343639778L)
    test(0L, -26151666524701044L, -240635881920542488L)
    test(-56L, 845191509087740151L, -14948325543844578L)
    test(0L, -162061855539686933L, -332451575304128502L)
    test(108L, 4253288611327629461L, 39367666671308052L)
    test(0L, -121341020863288159L, -2229209572918613645L)
    test(0L, 237137442026956209L, 1001116996845036907L)
    test(0L, -15661167300264588L, -116492150099255378L)
    test(-13L, 472755776244344837L, -34657749367478312L)
    test(0L, 61501256427799033L, 1117043674891618395L)
    test(8L, -816682331745911859L, -93013379356993805L)
    test(0L, -622502180540310102L, 861027865126430670L)
    test(-1L, -101326472862486775L, 53712510215619894L)
    test(0L, 23026094268252610L, 671734049378935892L)
    test(-1L, 2881176027443124423L, -2018901387779500606L)
    test(0L, 80880976030674953L, -148068225104828165L)
    test(31L, 939317388631011065L, 29631503184556902L)
    test(-19L, 2634188664743885501L, -132107816406700970L)
    test(0L, 28392039795497627L, -452518913994378813L)
    test(0L, -29885528417307733L, -3324874348720642133L)
    test(-2L, 591048418976612297L, -288205539284930378L)
    test(8L, 533073956562640706L, 65273067278234250L)
  }

  @Test def divisionByZero(): Unit = {
    @noinline def divNoInline(x: Long, y: Long): Long = x / y

    @inline def divInline(x: Long, y: Long): Long = x / y

    @inline def test(x: Long): Unit = {
      assertThrows(classOf[ArithmeticException], x / 0L)
      assertThrows(classOf[ArithmeticException], divNoInline(x, 0L))
      assertThrows(classOf[ArithmeticException], divInline(x, 0L))
    }

    test(0L)
    test(1L)
    test(43L)
    test(-3L)

    // Eligible for constant-folded by scalac itself
    assertThrows(classOf[ArithmeticException], 5L / 0L)
  }

  @Test def modulo(): Unit = {
    @inline def test(expected: Long, x: Long, y: Long): Unit = {
      assertEquals(expected, x % y)
      assertEquals(expected, hideFromOptimizer(x) % y)
      assertEquals(expected, x % hideFromOptimizer(y))
      assertEquals(expected, hideFromOptimizer(x) % hideFromOptimizer(y))
    }

    test(0L, IntMinVal, -1L)
    test(0L, IntMinVal, IntMaxValPlus1)
    test(0L, IntMaxValPlus1, -1L)
    test(0L, IntMaxValPlus1, IntMinVal)

    test(0L, MaxVal, -1L)
    test(0L, MinVal, 1L)
    test(0L, MinVal, -1L)

    test(9223372036854775807L, MaxVal, MinVal)
    test(0L, MaxVal, MaxVal)
    test(0L, MinVal, MinVal)
    test(-1L, MinVal, MaxVal)

    // int32, int32
    test(880L, 880L, -219594L)
    test(-27L, -49125L, 98L)
    test(-1194L, -1922504L, 4195L)
    test(3L, 3L, 7963L)
    test(-626L, -626L, -484628621L)
    test(11315L, 11315L, -3914076L)
    test(26241L, 15712341L, -1045740L)
    test(-507L, -855439L, 5213L)
    test(-259L, -101026259L, -500L)
    test(27720977L, 27720977L, -42317657L)
    test(1L, 25954L, -3L)
    test(6724180L, 338447650L, -8505730L)
    test(10488L, 23967L, -13479L)
    test(1L, 885202L, -3L)
    test(0L, 692795590L, -10L)
    test(-1L, -1L, 156L)
    test(388L, 388L, 189523294L)
    test(352L, 352L, -3257L)
    test(-9L, -9L, 14653L)
    test(-1L, -258745L, 8L)
    test(-21023L, -206976653L, 34321L)
    test(-1L, -1L, -971L)
    test(59L, 59L, 388L)
    test(0L, -7L, 1L)
    test(12L, 77L, 13L)
    test(224246L, 224246L, 719055L)
    test(-61296L, -61296L, -135723660L)
    test(549465L, 6897809L, 793543L)
    test(45L, 45L, 984210147L)
    test(0L, -64L, 1L)
    test(2L, 379611734L, 4L)
    test(0L, 0L, -263L)
    test(29L, 29L, -117L)
    test(24L, 245094L, -70L)
    test(0L, 0L, 5L)
    test(2L, 2L, 47787927L)
    test(-124L, -124L, -22714040L)
    test(412L, 412L, -17176L)
    test(-11860L, -11860L, 9506787L)
    test(-31L, -31L, -1544676L)
    test(-3L, -1990315281L, -7L)
    test(99L, 99L, -277L)
    test(-86L, -29227L, -161L)
    test(106L, 106L, -47032956L)
    test(18L, 18L, 510836179L)
    test(2L, 3543112L, 10L)
    test(534271L, 3547603L, -1506666L)
    test(-16361L, -16361L, 10637613L)
    test(8L, 606879016L, -16L)
    test(-1L, -1L, 46424570L)

    // int32, int53
    test(-3L, -3L, 4501768361L)
    test(-57756L, -57756L, 58918491782L)
    test(0L, 0L, 43125290916540L)
    test(423524L, 423524L, 70929482476L)
    test(38317L, 38317L, 105675177856L)
    test(60291L, 60291L, 244354846581L)
    test(1L, 1L, 5282610957902448L)
    test(296788L, 296788L, 223521545252L)
    test(-2005515L, -2005515L, 73346179491L)
    test(-179812L, -179812L, 22139703363330L)
    test(-3678L, -3678L, 1046277259413147L)
    test(-93867L, -93867L, 182758226138L)
    test(7600917L, 7600917L, 410489691890596L)
    test(300012L, 300012L, 435742913624L)
    test(-6347L, -6347L, 102640501950L)
    test(-41L, -41L, 1974472974044L)
    test(3425L, 3425L, 11303925732L)
    test(-25L, -25L, 111639285165193L)
    test(-8303L, -8303L, 8251985937002L)
    test(274032571L, 274032571L, 1096672203508L)
    test(-3L, -3L, 3132174934065L)
    test(-1124428L, -1124428L, 1459767596240L)
    test(-2L, -2L, 10844488562438L)
    test(1L, 1L, 8187934047L)
    test(107013504L, 107013504L, 13042506495L)
    test(4976822L, 4976822L, 9580026016302L)
    test(-1L, -1L, 180082425574L)
    test(80396L, 80396L, 59720539378L)
    test(937638L, 937638L, 115266897342L)
    test(756L, 756L, 7305399119272924L)
    test(5L, 5L, 91698197791401L)
    test(262831839L, 262831839L, 45669473529162L)
    test(-2146273993L, -2146273993L, 2755837895L)
    test(59799L, 59799L, 438440762347895L)
    test(-5347L, -5347L, 79274704127L)
    test(926L, 926L, 447563301356135L)
    test(1L, 1L, 5297346540373L)
    test(334L, 334L, 86480981154L)
    test(-61747L, -61747L, 6433018815983L)
    test(-1L, -1L, 341485018510162L)
    test(59605313L, 59605313L, 3132647545L)
    test(12267518L, 12267518L, 2441054592777302L)
    test(19230695L, 19230695L, 92038604353L)
    test(3950296L, 3950296L, 1047123350022L)
    test(503276L, 503276L, 6833559922L)
    test(30880536L, 30880536L, 219321124203675L)
    test(5659804L, 5659804L, 7864595535L)
    test(11882277L, 11882277L, 30308498427L)
    test(371783010L, 371783010L, 60134467254876L)
    test(840L, 840L, 470727040462L)

    // int32, big
    test(-267334310L, -267334310L, -578098228522141853L)
    test(57L, 57L, -43380562927871013L)
    test(30332L, 30332L, -386567995220878153L)
    test(187L, 187L, 35408332567946697L)
    test(-2L, -2L, 1678312288977938609L)
    test(-4252915L, -4252915L, 100866720606529501L)
    test(-2L, -2L, 416241567208853860L)
    test(10L, 10L, -343269240122331697L)
    test(-14L, -14L, 12765011240586127L)
    test(-20155233L, -20155233L, -109516752655403150L)
    test(-406L, -406L, 541589412425788400L)
    test(43L, 43L, 662582496523789435L)
    test(-4830L, -4830L, -12345854471231468L)
    test(-4271L, -4271L, -455809912113831520L)
    test(126L, 126L, -390003337959587581L)
    test(40445345L, 40445345L, 28987462963925530L)
    test(-1L, -1L, 122087865959677394L)
    test(-2123811L, -2123811L, 189636249314231355L)
    test(-25650126L, -25650126L, -69326722004354685L)
    test(30L, 30L, 681892781269423310L)
    test(2494211L, 2494211L, 93887941708414673L)
    test(100937174L, 100937174L, -321171615457988555L)
    test(901687L, 901687L, -2201437483664111307L)
    test(-422854L, -422854L, -424454611154691763L)
    test(2L, 2L, -42716550338485662L)
    test(-465211L, -465211L, -687950518813471351L)
    test(63L, 63L, 773357788582046200L)
    test(47L, 47L, -70955237701823702L)
    test(233450563L, 233450563L, 1621520427889937948L)
    test(-134912L, -134912L, -54171828449716468L)
    test(-95441L, -95441L, 69203073384780773L)
    test(-1160726496L, -1160726496L, -40549405630198558L)
    test(-502L, -502L, -1622331392079708275L)
    test(3313324L, 3313324L, 1899654667629672576L)
    test(-145L, -145L, 127665492386834182L)
    test(-19091L, -19091L, -746417751226485798L)
    test(-3331910L, -3331910L, 315702750057691225L)
    test(56622L, 56622L, 22416201665597149L)
    test(0L, 0L, 139463266603917347L)
    test(800L, 800L, 9875713007192617L)
    test(86949L, 86949L, -5946649264404445276L)
    test(10L, 10L, 4298700652581681596L)
    test(-333236L, -333236L, 537058674065071788L)
    test(-598L, -598L, -1417232804253270457L)
    test(-19L, -19L, 702749557261284541L)
    test(465328283L, 465328283L, -223584594313514269L)
    test(-25837L, -25837L, 37124400903503228L)
    test(-383633650L, -383633650L, 35485060063777451L)
    test(-66L, -66L, 1029153726012930463L)
    test(1676620L, 1676620L, 355475425885479146L)

    // int53 / int32
    test(15827410L, 14129525327L, -231372097L)
    test(15118L, 535478124622L, -20252L)
    test(11L, 309815810367L, 13L)
    test(42298679L, 15343123799L, -95630157L)
    test(17447610L, 127082894730L, -45315780L)
    test(0L, 1089230016351334L, -11L)
    test(51980L, 33057920873202L, 1489246L)
    test(2L, 6252840555450L, 8L)
    test(5125741L, 211719788413L, -34627848L)
    test(77691L, 91971976662285L, 149449L)
    test(521867604L, 2806186706724L, 671211684L)
    test(14579368L, 4273853776L, 177469767L)
    test(0L, 137176127796L, 1L)
    test(24027362L, 8425966166L, 33341027L)
    test(6792805L, 61762298457697L, -11334498L)
    test(9L, 7809058585409L, -10L)
    test(204L, 110067902189L, -235L)
    test(61089L, 2717761007L, 1618642L)
    test(289305533L, 2161231946023L, -321808286L)
    test(7272892L, 239618019295L, 15166197L)
    test(3L, 15433620448578L, 7L)
    test(12036L, 179964225338842L, -20591L)
    test(29L, 170743054973L, 48L)
    test(0L, 63933442317L, 1L)
    test(260441364L, 289183097958L, 1010219079L)
    test(3936541L, 138777709933L, -4427443L)
    test(183313645L, 3344958680351L, -273780418L)
    test(91783L, 2410446282098536L, -156677L)
    test(5L, 165936653941L, -8L)
    test(11214823L, 795868675507147L, -19368267L)
    test(75719L, 327928751549867L, 94212L)
    test(10941L, 236458995808L, 17599L)
    test(5331L, 501747583891L, -14942L)
    test(1L, 1018146484162351L, -2L)
    test(24400L, 125019445157368L, -37848L)
    test(95L, 4271993459155085L, -170L)
    test(9836L, 32301991436L, -10100L)
    test(30255783L, 52905400908L, -38454651L)
    test(417L, 19346043042L, 6825L)
    test(0L, 36027253440L, 2L)
    test(257L, 4209488274417L, -845L)
    test(82991L, 35239435579599L, 105392L)
    test(691256L, 8257556698L, 882238L)
    test(0L, 48993903540L, -20L)
    test(4L, 5301351495200050L, -13L)
    test(150L, 74179579248540L, 11033L)
    test(14L, 57030792056L, -23L)
    test(256216433L, 3216838357L, 740155481L)
    test(45583L, 15855664793359L, -63588L)
    test(459L, 6312015826290975L, -502L)

    // int53, int53
    test(6100144474L, 6100144474L, 1763937724960L)
    test(12301461237L, 7782416962849632L, 13976141337L)
    test(5641274328L, 5641274328L, 145356552798L)
    test(348750706398L, 348750706398L, 697512769240164L)
    test(5039243323L, 27190138897695L, 5337738004L)
    test(29273105L, 111580375427L, 5070504651L)
    test(8973135037L, 18510346432499L, 25240618414L)
    test(1915384404291L, 1915384404291L, 1322775745166193L)
    test(66563411320291L, 337238385612127L, 67668743572959L)
    test(75556878017L, 75556878017L, 315965468896632L)
    test(200222196095L, 200222196095L, 3226379400818411L)
    test(2821442827793L, 138954438973037L, 4861892719473L)
    test(2623090810L, 524628597533230L, 3760624884L)
    test(1012951716177L, 1012951716177L, 62946143599282L)
    test(2631016392226L, 2631016392226L, 12562543301042L)
    test(24502759655744L, 269716367153504L, 40868934582960L)
    test(19470049613L, 392647117165837L, 30944801948L)
    test(13597049958L, 13597049958L, 12189589505874L)
    test(191998352795L, 191998352795L, 3699380048895726L)
    test(61316766466L, 822043338636418L, 131223183568L)
    test(2833219350L, 19325725280274L, 5495703089L)
    test(167671681803991L, 167671681803991L, 609495034800021L)
    test(86115567547L, 86115567547L, 34925892478207L)
    test(100396132611L, 2270038478531L, 108482117296L)
    test(1722095012L, 192572302736L, 2891669814L)
    test(21689352006966L, 21689352006966L, 68016102680965L)
    test(1030709774291L, 1030709774291L, 33023840129568L)
    test(10652960238L, 473387766338L, 13220994460L)
    test(1645694049213L, 4867202831074L, 3221508781861L)
    test(204153882854L, 6851909241714L, 302170698130L)
    test(36056708963L, 36056708963L, 17883074988258L)
    test(28031441760L, 140488949988L, 56228754114L)
    test(6159596714L, 6159596714L, 4169026498867883L)
    test(184930003162138L, 4131898509072325L, 438552056212243L)
    test(22656551412L, 22656551412L, 113212339389675L)
    test(32219528416L, 1187328064845472L, 32429144688L)
    test(49836539309L, 419173707721801L, 92685508886L)
    test(143463766729516L, 748952389102980L, 302744311186732L)
    test(50357913499696L, 7025561694924070L, 1162533963570729L)
    test(242732351104L, 4198374281226L, 3955641930122L)
    test(127926935141L, 127926935141L, 82293613539627L)
    test(2012610771245L, 51496098028909L, 12370871814416L)
    test(57176122448L, 9374842408948L, 74541330292L)
    test(196672043680L, 196672043680L, 705905777378603L)
    test(230390222864L, 6102587199902L, 652466330782L)
    test(2669055575772L, 508650794854087L, 3693297366995L)
    test(902909663L, 34740183778L, 6767454823L)
    test(244160986772L, 454465802184020L, 339478057696L)
    test(162217586832L, 162217586832L, 378482848219509L)
    test(8558331816L, 8558331816L, 1038995034941793L)

    // int53, big
    test(16249759881L, 16249759881L, 3974504957130349699L)
    test(69801090825685L, 69801090825685L, 33609950826369033L)
    test(13150892233L, 13150892233L, -13352491823470351L)
    test(7408616170L, 7408616170L, 323012663481879067L)
    test(253783346903L, 253783346903L, -15596620746994050L)
    test(30972089554240L, 30972089554240L, -597395697936563678L)
    test(1359229988914L, 1359229988914L, -70151378460850387L)
    test(22776954844L, 22776954844L, 8338492573096515492L)
    test(2589224141516L, 2589224141516L, -582886159682980665L)
    test(2146722475297L, 2146722475297L, 54788046706768692L)
    test(2621531318650L, 2621531318650L, -691006344228255894L)
    test(4278483743L, 4278483743L, -7507719163850761352L)
    test(1038098064271L, 1038098064271L, -16401973284007851L)
    test(113168563974L, 113168563974L, -75157404508782419L)
    test(21796590416779L, 21796590416779L, 133448491453053159L)
    test(301847845516L, 301847845516L, -10696056474562902L)
    test(276953913912055L, 276953913912055L, 286740098556819600L)
    test(66738347538L, 66738347538L, 35225365194683174L)
    test(859025882164L, 859025882164L, -87272109905754421L)
    test(130253635110L, 130253635110L, -516770689141439465L)
    test(165643343437L, 165643343437L, 258119112451680666L)
    test(457118714799838L, 457118714799838L, -27186111829755469L)
    test(7223147609530411L, 7223147609530411L, 3263844584201112404L)
    test(47628957944688L, 47628957944688L, 14966006350759263L)
    test(1797578806720604L, 1797578806720604L, 14755661816785046L)
    test(344889987516L, 344889987516L, 1074257614097638658L)
    test(836850064448957L, 836850064448957L, 255240963022984568L)
    test(690921554798L, 690921554798L, 662819742471320650L)
    test(873327926453293L, 873327926453293L, 54527238454039994L)
    test(28722407527057L, 28722407527057L, 575601986231877219L)
    test(236635853491382L, 236635853491382L, -519622234960070382L)
    test(909765505249L, 909765505249L, -33816862537130552L)
    test(16180618491099L, 16180618491099L, -3235734480858584223L)
    test(132036592953L, 132036592953L, -226189298761468375L)
    test(9197248206L, 9197248206L, 254197295582276395L)
    test(2649490767787184L, 2649490767787184L, -195973906032954538L)
    test(4299787879680L, 4299787879680L, -48216847057476434L)
    test(92554360220420L, 92554360220420L, -498043957092554165L)
    test(138044092947625L, 138044092947625L, -15633878625202712L)
    test(851919394660L, 851919394660L, -473658874706870235L)
    test(3717788240566689L, 3717788240566689L, 11036376673790284L)
    test(30945796487505L, 30945796487505L, -68136163659478143L)
    test(5949003144209073L, 5949003144209073L, -35994306852615237L)
    test(17936864813251L, 17936864813251L, 133705877164730270L)
    test(37916212916L, 37916212916L, 2954580780708018654L)
    test(34796376878256L, 34796376878256L, -20676849320796038L)
    test(1047258769169L, 1047258769169L, 135053992753680793L)
    test(202109157226684L, 202109157226684L, -25513642512638919L)
    test(3788149908L, 3788149908L, 45352930034800570L)
    test(484613624593729L, 484613624593729L, -2861163538035533427L)

    // big, int32
    test(-3L, -1370393961416443737L, -6L)
    test(464005L, 3498020946037312230L, 825883L)
    test(34559370L, 11571581487268923L, 108493743L)
    test(-286379L, -2708099963766068459L, -506616L)
    test(-62L, -100064861483392562L, -206L)
    test(386945695L, 11246329772673651L, 1225551197L)
    test(270232L, 11888849560393702L, -291653L)
    test(277129L, 14906960692027984L, -29714535L)
    test(15L, 5449889234700691356L, -121L)
    test(1L, 148558486733610669L, 2L)
    test(46669L, 804275144639233391L, 129274L)
    test(-1508L, -1045956962716058568L, -31235L)
    test(0L, -1718682581097092450L, -1L)
    test(-1680L, -242610712566980878L, 2626L)
    test(53L, 49536209618101139L, 59L)
    test(-1699972L, -112595231960155440L, -2142359L)
    test(-200041L, -111202467621161165L, 483607L)
    test(-13123232L, -17543954451355937L, 428648085L)
    test(0L, -1577902156112015916L, 7L)
    test(-4528L, -1439263069294332652L, 5502L)
    test(-71L, -43275173539104061L, 110L)
    test(0L, 22222833743042817L, -1L)
    test(7872112L, 13138840034519224L, -29413816L)
    test(-37L, -15588082407912742L, -85L)
    test(14227L, 304145128647476139L, -14561L)
    test(-49L, -552172433790738993L, -256L)
    test(6101L, 67987053199001551L, 195494L)
    test(-6L, -23229285526738638L, 11L)
    test(-137624717L, -74919122060047792L, -168087095L)
    test(-3592L, -1593952962110360962L, 4765L)
    test(4335L, 756440483003859179L, -5564L)
    test(-271754L, -2567899230364692674L, -413908L)
    test(-361112L, -131350233252654878L, -496311L)
    test(-4L, -199379822739198927L, 29L)
    test(414436L, 13195486437163619L, 438221L)
    test(0L, -1290141875629267497L, 1L)
    test(-454L, -1191229212361459250L, -1237L)
    test(586891857L, 961830664768998345L, 707359548L)
    test(2L, 114848444499577496L, -3L)
    test(-24053960L, -375741920772048173L, 80229261L)
    test(-79944815L, -703102898231292331L, -983334452L)
    test(2600110L, 1915518801432622570L, 1035472980L)
    test(74L, 12822669827243498L, -148L)
    test(0L, 218533980803599700L, -1L)
    test(106L, 331433769486199795L, -563L)
    test(0L, -10261989598247925L, 1L)
    test(659L, 140012532054223616L, -729L)
    test(968L, 381558400809215180L, 13378L)
    test(920991L, 197728213235670981L, 922370L)
    test(2462152L, 28096912645232162L, -8405198L)

    // big, int53
    test(20393562716713L, 178525956958972613L, 78984762564715L)
    test(80530168971L, 120859823286244081L, 83815712363L)
    test(-48574599516617L, -746365263938012606L, 61694361357237L)
    test(-159605477404L, -28743647516819522L, 1777471270258L)
    test(-130067810337L, -921677148722623905L, 1626014442824L)
    test(-5545084389096L, -297533196331696083L, 10267717543131L)
    test(7729508244L, 1576350401918501173L, 50861495929L)
    test(-5425788258053L, -141792118906388348L, 12719717692485L)
    test(580541203635402L, 311841143885208818L, 1095988037611174L)
    test(-180987651967588L, -140226684536981914L, 246994174400378L)
    test(833309352355L, 5257458635234442787L, 1208023361952L)
    test(-22579028592974L, -47476342822552146L, 27864805516124L)
    test(9744525439149L, 736474376002594749L, 16812725583900L)
    test(-2843945215351L, -4166583957483287041L, 12900830149977L)
    test(-95865922570233L, -7254881016852281121L, 156363237944904L)
    test(134929788733543L, 4296825268368993527L, 137256910892546L)
    test(-207046992163300L, -60631062060489279L, 507764832506941L)
    test(-450506499492950L, -16279270902200926L, 545819462162344L)
    test(-21001025749L, -120715296107306033L, 683857870204L)
    test(110713754053227L, 11810000429759325L, 185702963106446L)
    test(-80504845900L, -66865102423090775L, 241806366625L)
    test(29361316939605L, 58425424162599923L, 193364446508809L)
    test(126332132307L, 495975906374412307L, 324618034880L)
    test(11750310041475L, 13098795247127361L, 14256040236477L)
    test(-2820995231L, -28082971156288335L, 2886317458L)
    test(-856087747000L, -7294235031903546523L, 2207709354381L)
    test(127372407516579L, 11875343380291413L, 279713594589877L)
    test(-95189779224694L, -201555894005618344L, 550439082585775L)
    test(-28210944743796L, -1037785570908536901L, 32835227336301L)
    test(4876153249L, 2839783057425459229L, 6407327394L)
    test(736189873568640L, 6875482986828220008L, 742172816253336L)
    test(-54624415368L, -118006778504953818L, 72596916825L)
    test(5102813362L, 39332119031229253L, 7602748813L)
    test(5244962969L, 86295194778571611L, 5401388374L)
    test(33647288201L, 16471371769530323L, 61350792699L)
    test(-16050338151L, -79135962419075376L, 28953323041L)
    test(12981486165522L, 228753608673200210L, 15736146614408L)
    test(2320531678876581L, 34527401002950741L, 3220686932407416L)
    test(-40680649487109L, -812780619607995673L, 53030140869014L)
    test(27897231688L, 1202898358452196048L, 64099076376L)
    test(-722136158L, -221468877749422237L, 3275821841L)
    test(-5898655866L, -9215248956777098454L, 68235756031L)
    test(763722238061482L, 2052506370068787562L, 2334178211411520L)
    test(347399715894725L, 514090742964385997L, 403886276138751L)
    test(82266055037649L, 192767004824633149L, 98559968680100L)
    test(898959150979L, 1158797429354770081L, 1436860683609L)
    test(-20037981041L, -37961764798073149L, 27673024823L)
    test(-72717556642L, -97162772410332351L, 272117928569L)
    test(-165925766094L, -247893552858557017L, 183293309603L)
    test(1479797283485806L, 4220161867656721306L, 4244146952085750L)

    // big, big
    test(883059337460449097L, 883059337460449097L, -1550174044384773417L)
    test(-9543412627974774L, -1596905473229600544L, -48101880624291690L)
    test(-10486616343740193L, -139303458961510665L, -11710622056160952L)
    test(81149513920264088L, 981926948276712668L, 300259144785482860L)
    test(9009882275485474L, 9009882275485474L, 79923769055871012L)
    test(-19932235341099321L, -506314354450506803L, 37414009162262114L)
    test(-12129109812415433L, -12129109812415433L, -2837741386438975528L)
    test(270784395103096803L, 270784395103096803L, 1612528131811460834L)
    test(83859782188007132L, 83859782188007132L, 3644002888077209617L)
    test(1746519510828363190L, 1746519510828363190L, 2111669249919511875L)
    test(860799297181306610L, 860799297181306610L, -1412005147950594882L)
    test(234265100281002846L, 2191676294256233782L, 489352798493807734L)
    test(-13539569498274996L, -4452079248766003228L, 23735506306244536L)
    test(91370955325389303L, 6249777914395455887L, 118430903059039742L)
    test(-16411784352803327L, -16411784352803327L, -20385458013414654L)
    test(-5887325254449551L, -22305484444692601L, -16418159190243050L)
    test(22270342034497702L, 22270342034497702L, -66419375563091088L)
    test(207601032925887705L, 666019080114845530L, 458418047188957825L)
    test(47277983711388430L, 47277983711388430L, -2109772615529102760L)
    test(-704328891862849825L, -704328891862849825L, 2276831382955067502L)
    test(-385964656907145510L, -1943847791817249050L, 389470783727525885L)
    test(-26138261192193017L, -369890880368811227L, 57292103196103035L)
    test(27655717698414314L, 27655717698414314L, -88623210967528211L)
    test(543130187522154L, 1012955677254479086L, -38938944117959882L)
    test(-11415718626796457L, -6587569366595715394L, -25004386494178399L)
    test(-487347157564673452L, -487347157564673452L, -3509587062480584104L)
    test(-783760630633017209L, -783760630633017209L, -2558190581855276594L)
    test(7102865471817366L, 851578697173205241L, -11259677756018505L)
    test(5421656277549904L, 1072836539808888926L, 28849050906252406L)
    test(50081603807435201L, 50081603807435201L, 3843187566851690054L)
    test(12229355184416833L, 12229355184416833L, 2960824918561572509L)
    test(-16093544120601009L, -232956518018373358L, -30980424842538907L)
    test(64957442391474035L, 64957442391474035L, -171662220355549567L)
    test(8955319424775302L, 23150711558820742L, -14195392134045440L)
    test(-44782815960861120L, -44782815960861120L, -7466833304991332587L)
    test(-17992712993945554L, -1092009147168492376L, -537008217087273411L)
    test(-7873560312580230L, -23378354844686757L, 15504794532106527L)
    test(2613971449376623L, 589867370672363331L, 12494753174957164L)
    test(-26919191832277496L, -1222102598859431176L, -74698962939197105L)
    test(-21166931130425933L, -139930636312903327L, 118763705182477394L)
    test(238558283776615026L, 238558283776615026L, 3989754685942986200L)
    test(-76394705346847293L, -216784999385471550L, 140390294038624257L)
    test(-9599424061286026L, -3941016588643634046L, -14835536470122068L)
    test(-328598693087048219L, -901033158038560664L, 572434464951512445L)
    test(-55762208105690701L, -55762208105690701L, 83369854153389342L)
    test(79579040298589706L, 2237037492842351339L, 102736116787798173L)
    test(-13418102631886498L, -502614740376610358L, -24459831887236193L)
    test(-21278129215907032L, -773733797545036687L, -68405060757193605L)
    test(2561606633654628L, 941995737089482432L, 234858532613956951L)
    test(-46319249618260476L, -46319249618260476L, 3256129778140263880L)
  }

  @Test def moduloByZero(): Unit = {
    @noinline def modNoInline(x: Long, y: Long): Long = x % y

    @inline def modInline(x: Long, y: Long): Long = x % y

    @inline def test(x: Long): Unit = {
      assertThrows(classOf[ArithmeticException], x % 0L)
      assertThrows(classOf[ArithmeticException], modNoInline(x, 0L))
      assertThrows(classOf[ArithmeticException], modInline(x, 0L))
    }

    test(0L)
    test(1L)
    test(43L)
    test(-3L)

    // Eligible for constant-folded by scalac itself
    assertThrows(classOf[ArithmeticException], 5L % 0L)
  }
}

object LongTest {

  case class HashTestBox(long: Long)

}
