// @flow

import type Wrapper from './wrapper'
import type VueWrapper from './vue-wrapper'
import { throwError } from 'shared/util'

export default class WrapperArray implements BaseWrapper {
  +wrappers: Array<Wrapper | VueWrapper>
  +length: number
  selector: Selector | void

  constructor(wrappers: Array<Wrapper | VueWrapper>) {
    const length = wrappers.length
    // $FlowIgnore
    Object.defineProperty(this, 'wrappers', {
      get: () => wrappers,
      set: () => throwError('wrapperArray.wrappers is read-only')
    })
    // $FlowIgnore
    Object.defineProperty(this, 'length', {
      get: () => length,
      set: () => throwError('wrapperArray.length is read-only')
    })
  }

  at(index: number): Wrapper | VueWrapper {
    const normalizedIndex = index < 0 ? this.length + index : index
    if (normalizedIndex > this.length - 1 || normalizedIndex < 0) {
      let error = `no item exists at ${index}`
      error += index < 0 ? ` (normalized to ${normalizedIndex})` : ''
      throwError(error)
    }
    return this.wrappers[normalizedIndex]
  }

  attributes(): void {
    this.throwErrorIfWrappersIsEmpty('attributes')

    throwError(
      `attributes must be called on a single wrapper, use ` +
        `at(i) to access a wrapper`
    )
  }

  classes(): void {
    this.throwErrorIfWrappersIsEmpty('classes')

    throwError(
      `classes must be called on a single wrapper, use ` +
        `at(i) to access a wrapper`
    )
  }

  contains(selector: Selector): boolean {
    this.throwErrorIfWrappersIsEmpty('contains')

    return this.wrappers.every(wrapper => wrapper.contains(selector))
  }

  exists(): boolean {
    return this.length > 0 && this.wrappers.every(wrapper => wrapper.exists())
  }

  filter(predicate: Function): WrapperArray {
    return new WrapperArray(this.wrappers.filter(predicate))
  }

  emitted(): void {
    this.throwErrorIfWrappersIsEmpty('emitted')

    throwError(
      `emitted must be called on a single wrapper, use ` +
        `at(i) to access a wrapper`
    )
  }

  emittedByOrder(): void {
    this.throwErrorIfWrappersIsEmpty('emittedByOrder')

    throwError(
      `emittedByOrder must be called on a single wrapper, ` +
        `use at(i) to access a wrapper`
    )
  }

  findAll(): void {
    this.throwErrorIfWrappersIsEmpty('findAll')

    throwError(
      `findAll must be called on a single wrapper, use ` +
        `at(i) to access a wrapper`
    )
  }

  find(): void {
    this.throwErrorIfWrappersIsEmpty('find')

    throwError(
      `find must be called on a single wrapper, use at(i) ` +
        `to access a wrapper`
    )
  }

  html(): void {
    this.throwErrorIfWrappersIsEmpty('html')

    throwError(
      `html must be called on a single wrapper, use at(i) ` +
        `to access a wrapper`
    )
  }

  is(selector: Selector): boolean {
    this.throwErrorIfWrappersIsEmpty('is')

    return this.wrappers.every(wrapper => wrapper.is(selector))
  }

  isEmpty(): boolean {
    this.throwErrorIfWrappersIsEmpty('isEmpty')

    return this.wrappers.every(wrapper => wrapper.isEmpty())
  }

  isVisible(): boolean {
    this.throwErrorIfWrappersIsEmpty('isVisible')

    return this.wrappers.every(wrapper => wrapper.isVisible())
  }

  isVueInstance(): boolean {
    this.throwErrorIfWrappersIsEmpty('isVueInstance')

    return this.wrappers.every(wrapper => wrapper.isVueInstance())
  }

  name(): void {
    this.throwErrorIfWrappersIsEmpty('name')

    throwError(
      `name must be called on a single wrapper, use at(i) ` +
        `to access a wrapper`
    )
  }

  overview(): void {
    this.throwErrorIfWrappersIsEmpty('overview()')

    throwError(
      `overview() must be called on a single wrapper, use at(i) ` +
        `to access a wrapper`
    )
  }

  props(): void {
    this.throwErrorIfWrappersIsEmpty('props')

    throwError(
      `props must be called on a single wrapper, use ` +
        `at(i) to access a wrapper`
    )
  }

  text(): void {
    this.throwErrorIfWrappersIsEmpty('text')

    throwError(
      `text must be called on a single wrapper, use at(i) ` +
        `to access a wrapper`
    )
  }

  throwErrorIfWrappersIsEmpty(method: string): void {
    if (this.wrappers.length === 0) {
      throwError(`${method} cannot be called on 0 items`)
    }
  }

  setData(data: Object): Promise<any> {
    this.throwErrorIfWrappersIsEmpty('setData')

    return Promise.all(this.wrappers.map(wrapper => wrapper.setData(data)))
  }

  setMethods(props: Object): void {
    this.throwErrorIfWrappersIsEmpty('setMethods')

    this.wrappers.forEach(wrapper => wrapper.setMethods(props))
  }

  setProps(props: Object): Promise<any> {
    this.throwErrorIfWrappersIsEmpty('setProps')

    return Promise.all(this.wrappers.map(wrapper => wrapper.setProps(props)))
  }

  setValue(value: any): Promise<any> {
    this.throwErrorIfWrappersIsEmpty('setValue')

    return Promise.all(this.wrappers.map(wrapper => wrapper.setValue(value)))
  }

  setChecked(checked: boolean = true): Promise<any> {
    this.throwErrorIfWrappersIsEmpty('setChecked')

    return Promise.all(
      this.wrappers.map(wrapper => wrapper.setChecked(checked))
    )
  }

  setSelected(): void {
    this.throwErrorIfWrappersIsEmpty('setSelected')

    throwError(
      `setSelected must be called on a single wrapper, ` +
        `use at(i) to access a wrapper`
    )
  }

  trigger(event: string, options: Object): Promise<any> {
    this.throwErrorIfWrappersIsEmpty('trigger')

    return Promise.all(
      this.wrappers.map(wrapper => wrapper.trigger(event, options))
    )
  }

  destroy(): void {
    this.throwErrorIfWrappersIsEmpty('destroy')

    this.wrappers.forEach(wrapper => wrapper.destroy())
  }
}
